package com.github.scalafanatic
package scalaalgorithms

import scala.collection.mutable.ArrayBuffer

object EditDistance {
  /**
    * Computes the Levenshtein distance between two strings using the "two matrix rows" approach.
    *
    * Reference: [[https://en.wikipedia.org/wiki/Levenshtein_distance]]
    */
  def compute(string1: String, string2: String): Int = {
    if (string1.isEmpty || string2.isEmpty) Math.max(string1.length, string2.length)
    else {
      // We initialize previousRow with values i = {0, 1, ..., string1.length} because this row is
      // matrix[0][i] (a.k.a. the edit distance from the empty string to every i in string1).
      var previousRow = ArrayBuffer(0 to string1.length: _*) // Using "to" instead of "until"
      var currentRow = ArrayBuffer.empty[Int]
      string2.toCharArray.zipWithIndex.foreach {
        case (letterFromString2, index2) =>
          currentRow = ArrayBuffer(index2 + 1)
          string1.toCharArray.zipWithIndex.foreach {
            case (letterFromString1, index1) =>
              if (letterFromString2 == letterFromString1) {
                // Copy the distance diagonally down from the previous row because it will not
                // change (these two letters are the same)
                currentRow.append(previousRow(index1))
              } else {
                /*
                  Appending to currentRow with the following will equate to performing the
                  corresponding operation.
                  ----------------------------------------------------------------------------------
                  previousRow(index1) =>
                      Substitute a letter (copy the distance diagonally down/right in the matrix)
                  previousRow(index1 + 1) =>
                      Delete a letter (copy the distance down in the matrix)
                  currentRow.last =>
                      Insert a letter (copy the distance right in the matrix)
                  ----------------------------------------------------------------------------------
                  We add 1 to the minimum of these 3 because the added distance incurred for all of
                  them is 1 (because this current letter is not the same as its counterpart in the
                  other string).
                 */
                val bestChoice =
                  List(previousRow(index1), previousRow(index1 + 1), currentRow.last).min
                currentRow.append(1 + bestChoice)
              }
          }
          previousRow = currentRow // Proceeding to the next row...
      }
      currentRow.last
    }
  }
}
