package com.github.scalafanatic
package scalaalgorithms

import scala.collection.mutable.ListBuffer

object WellOrderedStrings {
  val Alphabet: Array[Char] = Array(
    // Using an array for the alphabet allows for easy adjustment of the collation
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S',
    'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l',
    'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'
  )

  /**
    * Generates a list of all N-length strings that are "well-ordered" (i.e. alphabetically sorted)
    * based on some predefined alphabet.
    *
    * The formula for the number of strings generated is given by the binomial `(N+A-1, A-1)` where
    * `A` is the size of the alphabet and `N` is the length of the strings to be generated. This
    * leads to the [[https://en.wikipedia.org/wiki/Binomial_coefficient binomial coefficient]]
    * formula `(N+A-1)!/((N+A-1-(A-1))!(A-1)!)`, which simplifies to `(N+A-1)!/(N!(A-1)!)`. This
    * final formula is used in the asymptotic analysis of this algorithm.
    *
    * For more information about these formulae, refer to [[https://oeis.org/A017715]].
    */
  def generate(length: Int, alphabet: Array[Char]): List[String] = {
    val strings = ListBuffer.empty[String]
    val alphabetMap = alphabet.zipWithIndex.toMap // One-time upfront cost for efficient lookups
    def helper(current: String, lastLetterAdded: Option[Char]): Unit = {
      if (current.length == length) {
        strings.append(current)
      } else {
        // If we instead wanted to generate all possible strings of length N, we would assign
        // index = 0 below (regardless of what current is)
        val index = lastLetterAdded.fold(0)(alphabetMap.apply)
        for (i <- index until alphabet.length) {
          helper(current + alphabet(i), Some(alphabet(i)))
        }
      }
    }
    helper("", None)
    strings.toList // This is a constant-time conversion!
  }
}
