/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.river.mongodb;

import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.MongoClient;
import com.mongodb.MongoCursorNotFoundException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoSocketException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;
import com.mongodb.gridfs.GridFSFile;
import com.mongodb.util.JSONSerializers;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import org.bson.BSONObject;
import org.bson.BasicBSONObject;
import org.bson.types.ObjectId;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.collect.ImmutableList;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.ESLoggerFactory;
import org.elasticsearch.river.mongodb.CollectionSlurper;
import org.elasticsearch.river.mongodb.MongoDBRiver;
import org.elasticsearch.river.mongodb.MongoDBRiverDefinition;
import org.elasticsearch.river.mongodb.Operation;
import org.elasticsearch.river.mongodb.SharedContext;
import org.elasticsearch.river.mongodb.Status;
import org.elasticsearch.river.mongodb.Timestamp;
import org.elasticsearch.river.mongodb.util.MongoDBHelper;
import org.elasticsearch.river.mongodb.util.MongoDBRiverHelper;

class OplogSlurper
implements Runnable {
    private static final ESLogger logger = ESLoggerFactory.getLogger((String)OplogSlurper.class.getName());
    private final MongoDBRiverDefinition definition;
    private final SharedContext context;
    private final BasicDBObject findKeys;
    private final String gridfsOplogNamespace;
    private final String cmdOplogNamespace;
    private final ImmutableList<String> oplogOperations = ImmutableList.of((Object)"d", (Object)"ur", (Object)"u", (Object)"i", (Object)"c");
    private final Client esClient;
    private final MongoClient mongoClusterClient;
    private final MongoClient mongoShardClient;
    private Timestamp<?> timestamp;
    private final DB slurpedDb;
    private final DB oplogDb;
    private final DBCollection oplogCollection;
    private final DBCollection oplogRefsCollection;
    private final AtomicLong totalDocuments = new AtomicLong();

    public OplogSlurper(Timestamp<?> timestamp, MongoClient mongoClusterClient, MongoClient mongoShardClient, MongoDBRiverDefinition definition, SharedContext context, Client esClient) {
        this.timestamp = timestamp;
        this.definition = definition;
        this.context = context;
        this.esClient = esClient;
        this.mongoClusterClient = mongoClusterClient;
        this.mongoShardClient = mongoShardClient;
        this.findKeys = new BasicDBObject();
        this.gridfsOplogNamespace = String.valueOf(definition.getMongoOplogNamespace()) + ".files";
        this.cmdOplogNamespace = String.valueOf(definition.getMongoDb()) + "." + "$cmd";
        if (definition.getExcludeFields() != null) {
            for (String key : definition.getExcludeFields()) {
                this.findKeys.put(key, (Object)0);
            }
        } else if (definition.getIncludeFields() != null) {
            for (String key : definition.getIncludeFields()) {
                this.findKeys.put(key, (Object)1);
            }
        }
        this.oplogDb = mongoShardClient.getDB("local");
        this.oplogCollection = this.oplogDb.getCollection("oplog.rs");
        this.oplogRefsCollection = this.oplogDb.getCollection("oplog.refs");
        this.slurpedDb = mongoShardClient.getDB(definition.getMongoDb());
    }

    @Override
    public void run() {
        while (this.context.getStatus() == Status.RUNNING) {
            try {
                DBCursor cursor = null;
                try {
                    cursor = this.oplogCursor(this.timestamp);
                    if (cursor == null) {
                        cursor = this.processFullOplog();
                    }
                    while (cursor.hasNext()) {
                        DBObject item = cursor.next();
                        Object applied = item.get("a");
                        if (applied != null && !applied.equals(Boolean.TRUE)) {
                            logger.debug("Encountered oplog entry with a:false, ts:" + item.get("ts"), new Object[0]);
                            break;
                        }
                        this.timestamp = this.processOplogEntry(item, this.timestamp);
                    }
                    logger.debug("Before waiting for 500 ms", new Object[0]);
                    Thread.sleep(500L);
                }
                finally {
                    if (cursor != null) {
                        logger.trace("Closing oplog cursor", new Object[0]);
                        cursor.close();
                    }
                }
            }
            catch (SlurperException e) {
                logger.error("Exception in slurper", (Throwable)e, new Object[0]);
                Thread.currentThread().interrupt();
                return;
            }
            catch (MongoInterruptedException | InterruptedException throwable) {
                logger.info("river-mongodb slurper interrupted", new Object[0]);
                Thread.currentThread().interrupt();
                return;
            }
            catch (MongoCursorNotFoundException | MongoSocketException | MongoTimeoutException e) {
                logger.info("Oplog tailing - {} - {}. Will retry.", new Object[]{e.getClass().getSimpleName(), e.getMessage()});
                logger.debug("Total documents inserted so far by river {}: {}", new Object[]{this.definition.getRiverName(), this.totalDocuments.get()});
                try {
                    Thread.sleep(10000L);
                }
                catch (InterruptedException interruptedException) {
                    logger.info("river-mongodb slurper interrupted", new Object[0]);
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            catch (Exception e) {
                logger.error("Exception while looping in cursor", (Throwable)e, new Object[0]);
                Thread.currentThread().interrupt();
                return;
            }
        }
        logger.info("Slurper is stopping. River has status {}", new Object[]{this.context.getStatus()});
    }

    protected boolean riverHasIndexedFromOplog() {
        return MongoDBRiver.getLastTimestamp(this.esClient, this.definition) != null;
    }

    protected boolean isIndexEmpty() {
        return MongoDBRiver.getIndexCount(this.esClient, this.definition) == 0L;
    }

    private Timestamp<?> getCurrentOplogTimestamp() {
        Throwable throwable = null;
        Object var2_3 = null;
        try (DBCursor cursor = this.oplogCollection.find().sort((DBObject)new BasicDBObject("$natural", (Object)-1)).limit(1);){
            return Timestamp.on(cursor.next());
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private DBCursor processFullOplog() throws InterruptedException, SlurperException {
        Timestamp<?> currentTimestamp = this.getCurrentOplogTimestamp();
        return this.oplogCursor(currentTimestamp);
    }

    private Timestamp<?> processOplogEntry(DBObject entry, Timestamp<?> startTimestamp) throws InterruptedException {
        this.flattenOps(entry);
        if (!this.isValidOplogEntry(entry, startTimestamp)) {
            return startTimestamp;
        }
        Operation operation = Operation.fromString(entry.get("op").toString());
        String namespace = entry.get("ns").toString();
        String collection = null;
        Timestamp<?> oplogTimestamp = Timestamp.on(entry);
        DBObject object = (DBObject)entry.get("o");
        if (this.definition.isImportAllCollections()) {
            if (namespace.startsWith(this.definition.getMongoDb()) && !namespace.equals(this.cmdOplogNamespace)) {
                collection = this.getCollectionFromNamespace(namespace);
            }
        } else {
            collection = this.definition.getMongoCollection();
        }
        if (namespace.equals(this.cmdOplogNamespace)) {
            if (object.containsField("drop")) {
                operation = Operation.DROP_COLLECTION;
                if (this.definition.isImportAllCollections() && (collection = object.get("drop").toString()).startsWith("tmp.mr.")) {
                    return startTimestamp;
                }
            }
            if (object.containsField("dropDatabase")) {
                operation = Operation.DROP_DATABASE;
            }
        }
        logger.trace("namespace: {} - operation: {}", new Object[]{namespace, operation});
        if (namespace.equals("admin.$cmd") && operation == Operation.COMMAND) {
            this.processAdminCommandOplogEntry(entry, startTimestamp);
            return startTimestamp;
        }
        if (logger.isTraceEnabled()) {
            String deserialized = object.toString();
            if (deserialized.length() < 400) {
                logger.trace("MongoDB object deserialized: {}", new Object[]{deserialized});
            } else {
                logger.trace("MongoDB object deserialized is {} characters long", new Object[]{deserialized.length()});
            }
            logger.trace("collection: {}", new Object[]{collection});
            logger.trace("oplog entry - namespace [{}], operation [{}]", new Object[]{namespace, operation});
            if (deserialized.length() < 400) {
                logger.trace("oplog processing item {}", new Object[]{entry});
            }
        }
        String objectId = this.getObjectIdFromOplogEntry(entry);
        if (operation == Operation.DELETE) {
            if (object.containsField("_id")) {
                if (object.keySet().size() > 1) {
                    object = new BasicDBObject("_id", (Object)objectId);
                    entry.put("o", (Object)object);
                }
            } else {
                throw new NullPointerException("_id");
            }
        }
        if (this.definition.isMongoGridFS() && namespace.endsWith(".files") && (operation == Operation.INSERT || operation == Operation.UPDATE)) {
            if (objectId == null) {
                throw new NullPointerException("_id");
            }
            GridFS grid = new GridFS(this.mongoShardClient.getDB(this.definition.getMongoDb()), collection);
            GridFSDBFile file = grid.findOne(new ObjectId(objectId));
            if (file != null) {
                logger.trace("Caught file: {} - {}", new Object[]{file.getId(), file.getFilename()});
                object = file;
            } else {
                logger.error("Cannot find file from id: {}", new Object[]{objectId});
            }
        }
        if (object instanceof GridFSDBFile) {
            if (objectId == null) {
                throw new NullPointerException("_id");
            }
            if (logger.isTraceEnabled()) {
                logger.trace("Add attachment: {}", new Object[]{objectId});
            }
            this.addToStream(operation, oplogTimestamp, this.applyFieldFilter(object), collection);
        } else if (operation == Operation.UPDATE) {
            DBObject update = (DBObject)entry.get("o2");
            logger.trace("Updated item: {}", new Object[]{update});
            this.addQueryToStream(operation, oplogTimestamp, update, collection);
        } else if (operation == Operation.INSERT) {
            this.addInsertToStream(oplogTimestamp, this.applyFieldFilter(object), collection);
        } else {
            this.addToStream(operation, oplogTimestamp, this.applyFieldFilter(object), collection);
        }
        return oplogTimestamp;
    }

    private void flattenOps(DBObject entry) {
        Object ops;
        Object ref = entry.removeField("ref");
        Object object = ops = ref == null ? entry.removeField("ops") : this.getRefOps(ref);
        if (ops != null) {
            try {
                for (DBObject op : (List)ops) {
                    DBObject object2;
                    String operation = (String)op.get("op");
                    if (operation.equals("c") && (object2 = (DBObject)op.get("o")).containsField("create")) continue;
                    entry.putAll((BSONObject)op);
                }
            }
            catch (ClassCastException e) {
                logger.error(e.toString(), (Throwable)e, new Object[0]);
            }
        }
    }

    private Object getRefOps(Object ref) {
        BasicDBObject query = new BasicDBObject("_id", (Object)new BasicDBObject("$gte", (Object)new BasicDBObject("oid", ref)));
        DBObject oplog = this.oplogRefsCollection.findOne((DBObject)query);
        return oplog == null ? null : oplog.get("ops");
    }

    private void processAdminCommandOplogEntry(DBObject entry, Timestamp<?> startTimestamp) throws InterruptedException {
        String to;
        if (logger.isTraceEnabled()) {
            logger.trace("processAdminCommandOplogEntry - [{}]", new Object[]{entry});
        }
        DBObject object = (DBObject)entry.get("o");
        if (this.definition.isImportAllCollections() && object.containsField("renameCollection") && object.containsField("to") && (to = object.get("to").toString()).startsWith(this.definition.getMongoDb())) {
            String newCollection = this.getCollectionFromNamespace(to);
            DBCollection coll = this.slurpedDb.getCollection(newCollection);
            CollectionSlurper importer = new CollectionSlurper(this.timestamp, this.mongoClusterClient, this.definition, this.context, this.esClient);
            importer.importCollection(coll);
        }
    }

    private String getCollectionFromNamespace(String namespace) {
        if (namespace.startsWith(String.valueOf(this.definition.getMongoDb()) + '.')) {
            return namespace.substring(this.definition.getMongoDb().length() + 1);
        }
        logger.error("Cannot get collection from namespace [{}]", new Object[]{namespace});
        return null;
    }

    private boolean isValidOplogEntry(DBObject entry, Timestamp<?> startTimestamp) {
        Timestamp<?> oplogTimestamp;
        if (!entry.containsField("op")) {
            logger.trace("[Empty Oplog Entry] - can be ignored. {}", new Object[]{JSONSerializers.getStrict().serialize((Object)entry)});
            return false;
        }
        if ("n".equals(entry.get("op"))) {
            logger.trace("[No-op Oplog Entry] - can be ignored. {}", new Object[]{JSONSerializers.getStrict().serialize((Object)entry)});
            return false;
        }
        String namespace = (String)entry.get("ns");
        if (entry.containsField("fromMigrate") && ((BasicBSONObject)entry).getBoolean("fromMigrate")) {
            logger.trace("[Invalid Oplog Entry] - from migration or sharding operation. Can be ignored. {}", new Object[]{JSONSerializers.getStrict().serialize((Object)entry)});
            return false;
        }
        if (namespace.endsWith(".chunks")) {
            return false;
        }
        if (startTimestamp != null && Timestamp.compare(oplogTimestamp = Timestamp.on(entry), startTimestamp) < 0) {
            logger.error("[Invalid Oplog Entry] - entry timestamp [{}] before startTimestamp [{}]", new Object[]{JSONSerializers.getStrict().serialize((Object)entry), startTimestamp});
            return false;
        }
        boolean validNamespace = false;
        if (this.definition.isMongoGridFS()) {
            validNamespace = this.gridfsOplogNamespace.equals(namespace);
        } else {
            if (this.definition.isImportAllCollections()) {
                if (namespace.startsWith(this.definition.getMongoDb()) && !namespace.startsWith(String.valueOf(this.definition.getMongoDb()) + ".tmp.mr")) {
                    validNamespace = true;
                }
            } else if (this.definition.getMongoOplogNamespace().equals(namespace)) {
                validNamespace = true;
            }
            if (this.cmdOplogNamespace.equals(namespace)) {
                validNamespace = true;
            }
            if ("admin.$cmd".equals(namespace)) {
                validNamespace = true;
            }
        }
        if (!validNamespace) {
            logger.trace("[Invalid Oplog Entry] - namespace [{}] is not valid", new Object[]{namespace});
            return false;
        }
        String operation = (String)entry.get("op");
        if (!this.oplogOperations.contains((Object)operation)) {
            logger.trace("[Invalid Oplog Entry] - operation [{}] is not valid", new Object[]{operation});
            return false;
        }
        if (this.definition.getMongoOplogFilter() != null) {
            DBObject object = (DBObject)entry.get("o");
            BasicDBObject filter = this.definition.getMongoOplogFilter();
            if (!this.filterMatch((DBObject)filter, object)) {
                logger.trace("[Invalid Oplog Entry] - filter [{}] does not match object [{}]", new Object[]{filter, object});
                return false;
            }
        }
        return true;
    }

    private boolean filterMatch(DBObject filter, DBObject object) {
        for (String key : filter.keySet()) {
            if (!object.containsField(key)) {
                return false;
            }
            if (filter.get(key).equals(object.get(key))) continue;
            return false;
        }
        return true;
    }

    private DBObject applyFieldFilter(DBObject object) {
        if (object instanceof GridFSFile) {
            GridFSFile file = (GridFSFile)object;
            DBObject metadata = file.getMetaData();
            if (metadata != null) {
                file.setMetaData(this.applyFieldFilter(metadata));
            }
        } else {
            object = MongoDBHelper.applyExcludeFields(object, this.definition.getExcludeFields());
            object = MongoDBHelper.applyIncludeFields(object, this.definition.getIncludeFields());
        }
        return object;
    }

    private String getObjectIdFromOplogEntry(DBObject entry) {
        DBObject object;
        if (entry.containsField("o") && (object = (DBObject)entry.get("o")).containsField("_id")) {
            return object.get("_id").toString();
        }
        if (entry.containsField("o2") && (object = (DBObject)entry.get("o2")).containsField("_id")) {
            return object.get("_id").toString();
        }
        return null;
    }

    private DBCursor oplogCursor(Timestamp<?> time) throws SlurperException {
        DBObject indexFilter = time.getOplogFilter();
        if (indexFilter == null) {
            return null;
        }
        int options = 58;
        DBCursor cursor = this.oplogCollection.find(indexFilter).setOptions(options);
        if (indexFilter.containsField("_id")) {
            cursor = cursor.hint("_id_");
        }
        this.isRiverStale(cursor, time);
        return cursor;
    }

    private void isRiverStale(DBCursor cursor, Timestamp<?> time) throws SlurperException {
        if (cursor == null || time == null) {
            return;
        }
        if (this.definition.getInitialTimestamp() != null && time.equals(this.definition.getInitialTimestamp())) {
            return;
        }
        DBObject entry = cursor.next();
        Timestamp<?> oplogTimestamp = Timestamp.on(entry);
        if (!time.equals(oplogTimestamp)) {
            MongoDBRiverHelper.setRiverStatus(this.esClient, this.definition.getRiverName(), Status.RIVER_STALE);
            throw new SlurperException("River out of sync with oplog.rs collection");
        }
    }

    private void addQueryToStream(Operation operation, Timestamp<?> currentTimestamp, DBObject update, String collection) throws InterruptedException {
        if (logger.isTraceEnabled()) {
            logger.trace("addQueryToStream - operation [{}], currentTimestamp [{}], update [{}]", new Object[]{operation, currentTimestamp, update});
        }
        if (collection == null) {
            for (String name : this.slurpedDb.getCollectionNames()) {
                DBCollection slurpedCollection = this.slurpedDb.getCollection(name);
                this.addQueryToStream(operation, currentTimestamp, update, name, slurpedCollection);
            }
        } else {
            DBCollection slurpedCollection = this.slurpedDb.getCollection(collection);
            this.addQueryToStream(operation, currentTimestamp, update, collection, slurpedCollection);
        }
    }

    private void addQueryToStream(Operation operation, Timestamp<?> currentTimestamp, DBObject update, String collection, DBCollection slurpedCollection) throws InterruptedException {
        Throwable throwable = null;
        Object var7_8 = null;
        try (DBCursor cursor = slurpedCollection.find(update, (DBObject)this.findKeys);){
            for (DBObject item : cursor) {
                this.addToStream(operation, currentTimestamp, item, collection);
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private String addInsertToStream(Timestamp<?> currentTimestamp, DBObject data, String collection) throws InterruptedException {
        this.totalDocuments.incrementAndGet();
        this.addToStream(Operation.INSERT, currentTimestamp, data, collection);
        if (data == null) {
            return null;
        }
        return data.containsField("_id") ? data.get("_id").toString() : null;
    }

    private void addToStream(Operation operation, Timestamp<?> currentTimestamp, DBObject data, String collection) throws InterruptedException {
        if (logger.isTraceEnabled()) {
            String dataString = data.toString();
            if (dataString.length() > 400) {
                logger.trace("addToStream - operation [{}], currentTimestamp [{}], data (_id:[{}], serialized length:{}), collection [{}]", new Object[]{operation, currentTimestamp, data.get("_id"), dataString.length(), collection});
            } else {
                logger.trace("addToStream - operation [{}], currentTimestamp [{}], data [{}], collection [{}]", new Object[]{operation, currentTimestamp, dataString, collection});
            }
        }
        if (operation == Operation.DROP_DATABASE) {
            logger.info("addToStream - Operation.DROP_DATABASE, currentTimestamp [{}], data [{}], collection [{}]", new Object[]{currentTimestamp, data, collection});
            if (this.definition.isImportAllCollections()) {
                for (String name : this.slurpedDb.getCollectionNames()) {
                    logger.info("addToStream - isImportAllCollections - Operation.DROP_DATABASE, currentTimestamp [{}], data [{}], collection [{}]", new Object[]{currentTimestamp, data, name});
                    this.context.getStream().put(new MongoDBRiver.QueueEntry(currentTimestamp, Operation.DROP_COLLECTION, data, name));
                }
            } else {
                this.context.getStream().put(new MongoDBRiver.QueueEntry(currentTimestamp, Operation.DROP_COLLECTION, data, collection));
            }
        } else {
            this.context.getStream().put(new MongoDBRiver.QueueEntry(currentTimestamp, operation, data, collection));
        }
    }

    class SlurperException
    extends Exception {
        private static final long serialVersionUID = 1L;

        SlurperException(String message) {
            super(message);
        }
    }
}

