/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.river.mongodb;

import com.google.common.base.Preconditions;
import com.mongodb.BasicDBObject;
import com.mongodb.DB;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.MongoClient;
import com.mongodb.MongoCursorNotFoundException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoSocketException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;
import com.mongodb.gridfs.GridFSFile;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import org.bson.BasicBSONObject;
import org.bson.types.ObjectId;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.collect.ImmutableList;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.ESLoggerFactory;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.river.mongodb.MongoDBRiver;
import org.elasticsearch.river.mongodb.MongoDBRiverDefinition;
import org.elasticsearch.river.mongodb.Operation;
import org.elasticsearch.river.mongodb.SharedContext;
import org.elasticsearch.river.mongodb.Status;
import org.elasticsearch.river.mongodb.Timestamp;
import org.elasticsearch.river.mongodb.util.MongoDBHelper;
import org.elasticsearch.river.mongodb.util.MongoDBRiverHelper;

class CollectionSlurper
implements Runnable {
    private static final ESLogger logger = ESLoggerFactory.getLogger((String)CollectionSlurper.class.getName());
    private final MongoDBRiverDefinition definition;
    private final SharedContext context;
    private final Client esClient;
    private final MongoClient mongoClient;
    private Timestamp<?> timestamp;
    private final DB slurpedDb;
    private final AtomicLong totalDocuments = new AtomicLong();

    public CollectionSlurper(Timestamp<?> timestamp, MongoClient mongoClient, MongoDBRiverDefinition definition, SharedContext context, Client esClient) {
        this.timestamp = timestamp;
        this.definition = definition;
        this.context = context;
        this.esClient = esClient;
        this.mongoClient = mongoClient;
        this.slurpedDb = mongoClient.getDB(definition.getMongoDb());
    }

    @Override
    public void run() {
        if (this.definition.isSkipInitialImport() || this.definition.getInitialTimestamp() != null) {
            logger.info("Skip initial import from collection {}", new Object[]{this.definition.getMongoCollection()});
        } else if (this.riverHasIndexedFromOplog()) {
            logger.trace("Initial import already completed.", new Object[0]);
        } else {
            try {
                if (!this.isIndexEmpty()) {
                    MongoDBRiverHelper.setRiverStatus(this.esClient, this.definition.getRiverName(), Status.INITIAL_IMPORT_FAILED);
                    return;
                }
                if (this.definition.isImportAllCollections()) {
                    for (String name : this.slurpedDb.getCollectionNames()) {
                        if (name.length() >= 7 && name.substring(0, 7).equals("system.")) continue;
                        DBCollection collection = this.slurpedDb.getCollection(name);
                        this.importCollection(collection);
                    }
                } else {
                    DBCollection collection = this.slurpedDb.getCollection(this.definition.getMongoCollection());
                    this.importCollection(collection);
                }
                logger.debug("Before waiting for 500 ms", new Object[0]);
                Thread.sleep(500L);
            }
            catch (MongoInterruptedException | InterruptedException throwable) {
                logger.info("river-mongodb slurper interrupted", new Object[0]);
                Thread.currentThread().interrupt();
                return;
            }
            catch (MongoCursorNotFoundException | MongoSocketException | MongoTimeoutException e) {
                logger.info("Oplog tailing - {} - {}. Will retry.", new Object[]{e.getClass().getSimpleName(), e.getMessage()});
                logger.debug("Total documents inserted so far by river {}: {}", new Object[]{this.definition.getRiverName(), this.totalDocuments.get()});
                try {
                    Thread.sleep(10000L);
                }
                catch (InterruptedException interruptedException) {
                    logger.info("river-mongodb slurper interrupted", new Object[0]);
                    Thread.currentThread().interrupt();
                    return;
                }
            }
            catch (Exception e) {
                logger.error("Exception while looping in cursor", (Throwable)e, new Object[0]);
                Thread.currentThread().interrupt();
                return;
            }
        }
    }

    protected boolean riverHasIndexedFromOplog() {
        return MongoDBRiver.getLastTimestamp(this.esClient, this.definition) != null;
    }

    protected boolean isIndexEmpty() {
        return MongoDBRiver.getIndexCount(this.esClient, this.definition) == 0L;
    }

    protected void importCollection(DBCollection collection) throws InterruptedException {
        logger.info("MongoDBRiver is beginning initial import of " + collection.getFullName(), new Object[0]);
        boolean inProgress = true;
        String lastId = null;
        while (inProgress) {
            DBCursor cursor = null;
            try {
                try {
                    if (this.definition.isDisableIndexRefresh()) {
                        this.updateIndexRefresh(this.definition.getIndexName(), -1L);
                    }
                    if (!this.definition.isMongoGridFS()) {
                        if (logger.isTraceEnabled()) {
                            logger.trace("Collection {} - count: {}", new Object[]{collection.getName(), collection.count()});
                        }
                        long count = 0L;
                        cursor = collection.find((DBObject)this.getFilterForInitialImport(this.definition.getMongoCollectionFilter(), lastId)).sort((DBObject)new BasicDBObject("_id", (Object)1));
                        while (cursor.hasNext() && this.context.getStatus() == Status.RUNNING) {
                            DBObject object = cursor.next();
                            ++count;
                            if (cursor.hasNext()) {
                                lastId = this.addInsertToStream(null, this.applyFieldFilter(object), collection.getName());
                                continue;
                            }
                            logger.debug("Last entry for initial import of {} - add timestamp: {}", new Object[]{collection.getFullName(), this.timestamp});
                            lastId = this.addInsertToStream(this.timestamp, this.applyFieldFilter(object), collection.getName());
                        }
                        inProgress = false;
                        logger.info("Number of documents indexed in initial import of {}: {}", new Object[]{collection.getFullName(), count});
                    } else {
                        GridFS grid = new GridFS(this.mongoClient.getDB(this.definition.getMongoDb()), this.definition.getMongoCollection());
                        cursor = grid.getFileList();
                        while (cursor.hasNext()) {
                            DBObject object = cursor.next();
                            if (!(object instanceof GridFSDBFile)) continue;
                            GridFSDBFile file = grid.findOne(new ObjectId(object.get("_id").toString()));
                            if (cursor.hasNext()) {
                                lastId = this.addInsertToStream(null, (DBObject)file);
                                continue;
                            }
                            logger.debug("Last entry for initial import of {} - add timestamp: {}", new Object[]{collection.getFullName(), this.timestamp});
                            lastId = this.addInsertToStream(this.timestamp, (DBObject)file);
                        }
                        inProgress = false;
                    }
                }
                catch (MongoCursorNotFoundException | MongoSocketException | MongoTimeoutException e) {
                    logger.info("Initial import - {} - {}. Will retry.", new Object[]{e.getClass().getSimpleName(), e.getMessage()});
                    logger.debug("Total documents inserted so far by river {}: {}", new Object[]{this.definition.getRiverName(), this.totalDocuments.get()});
                    Thread.sleep(10000L);
                    if (cursor != null) {
                        logger.trace("Closing initial import cursor", new Object[0]);
                        cursor.close();
                    }
                    if (!this.definition.isDisableIndexRefresh()) continue;
                    this.updateIndexRefresh(this.definition.getIndexName(), TimeValue.timeValueSeconds((long)1L));
                    continue;
                }
            }
            catch (Throwable throwable) {
                if (cursor != null) {
                    logger.trace("Closing initial import cursor", new Object[0]);
                    cursor.close();
                }
                if (this.definition.isDisableIndexRefresh()) {
                    this.updateIndexRefresh(this.definition.getIndexName(), TimeValue.timeValueSeconds((long)1L));
                }
                throw throwable;
            }
            if (cursor != null) {
                logger.trace("Closing initial import cursor", new Object[0]);
                cursor.close();
            }
            if (!this.definition.isDisableIndexRefresh()) continue;
            this.updateIndexRefresh(this.definition.getIndexName(), TimeValue.timeValueSeconds((long)1L));
        }
    }

    private BasicDBObject getFilterForInitialImport(BasicDBObject filter, String id) {
        Preconditions.checkNotNull((Object)filter);
        if (id == null) {
            return filter;
        }
        BasicDBObject idFilter = new BasicDBObject("_id", (Object)new BasicBSONObject("$gt", (Object)id));
        if (filter.equals((Object)new BasicDBObject())) {
            return idFilter;
        }
        return new BasicDBObject("$and", (Object)ImmutableList.of((Object)filter, (Object)idFilter));
    }

    private void updateIndexRefresh(String name, Object value) {
        this.esClient.admin().indices().prepareUpdateSettings(new String[]{name}).setSettings((Map)ImmutableMap.of((Object)"index.refresh_interval", (Object)value)).get();
    }

    private DBObject applyFieldFilter(DBObject object) {
        if (object instanceof GridFSFile) {
            GridFSFile file = (GridFSFile)object;
            DBObject metadata = file.getMetaData();
            if (metadata != null) {
                file.setMetaData(this.applyFieldFilter(metadata));
            }
        } else {
            object = MongoDBHelper.applyExcludeFields(object, this.definition.getExcludeFields());
            object = MongoDBHelper.applyIncludeFields(object, this.definition.getIncludeFields());
        }
        return object;
    }

    private String addInsertToStream(Timestamp<?> currentTimestamp, DBObject data) throws InterruptedException {
        return this.addInsertToStream(currentTimestamp, data, this.definition.getMongoCollection());
    }

    private String addInsertToStream(Timestamp<?> currentTimestamp, DBObject data, String collection) throws InterruptedException {
        this.totalDocuments.incrementAndGet();
        this.addToStream(Operation.INSERT, currentTimestamp, data, collection);
        if (data == null) {
            return null;
        }
        return data.containsField("_id") ? data.get("_id").toString() : null;
    }

    private void addToStream(Operation operation, Timestamp<?> currentTimestamp, DBObject data, String collection) throws InterruptedException {
        if (logger.isTraceEnabled()) {
            String dataString = data.toString();
            if (dataString.length() > 400) {
                logger.trace("addToStream - operation [{}], currentTimestamp [{}], data (_id:[{}], serialized length:{}), collection [{}]", new Object[]{operation, currentTimestamp, data.get("_id"), dataString.length(), collection});
            } else {
                logger.trace("addToStream - operation [{}], currentTimestamp [{}], data [{}], collection [{}]", new Object[]{operation, currentTimestamp, dataString, collection});
            }
        }
        if (operation == Operation.DROP_DATABASE) {
            logger.info("addToStream - Operation.DROP_DATABASE, currentTimestamp [{}], data [{}], collection [{}]", new Object[]{currentTimestamp, data, collection});
            if (this.definition.isImportAllCollections()) {
                for (String name : this.slurpedDb.getCollectionNames()) {
                    logger.info("addToStream - isImportAllCollections - Operation.DROP_DATABASE, currentTimestamp [{}], data [{}], collection [{}]", new Object[]{currentTimestamp, data, name});
                    this.context.getStream().put(new MongoDBRiver.QueueEntry(currentTimestamp, Operation.DROP_COLLECTION, data, name));
                }
            } else {
                this.context.getStream().put(new MongoDBRiver.QueueEntry(currentTimestamp, Operation.DROP_COLLECTION, data, collection));
            }
        } else {
            this.context.getStream().put(new MongoDBRiver.QueueEntry(currentTimestamp, operation, data, collection));
        }
    }
}

