/*
 * Copyright 2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.github.nosan.boot.test.autoconfigure.cassandra;

import java.net.InetAddress;

import com.datastax.driver.core.Cluster;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.cassandra.CassandraAutoConfiguration;
import org.springframework.boot.autoconfigure.cassandra.ClusterBuilderCustomizer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnSingleCandidate;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;

import com.github.nosan.embedded.cassandra.Cassandra;
import com.github.nosan.embedded.cassandra.Settings;
import com.github.nosan.embedded.cassandra.test.TestCassandra;

/**
 * {@link EnableAutoConfiguration Auto-configuration} for a test Cassandra.
 *
 * @author Dmytro Nosan
 * @since 1.0.2
 */
@Configuration
@AutoConfigureBefore(CassandraAutoConfiguration.class)
@ConditionalOnClass({Cassandra.class, Cluster.class})
public class EmbeddedCassandraTestAutoConfiguration {

	/**
	 * Configuration that overrides {@link Cluster} configuration via {@link ClusterBuilderCustomizer} bean using
	 * {@link TestCassandra} bean properties.
	 */
	@Configuration
	@ConditionalOnClass(TestCassandra.class)
	@ConditionalOnSingleCandidate(TestCassandra.class)
	static class TestCassandraClusterConfiguration {

		@Bean
		@Order(0)
		public ClusterBuilderCustomizer cassandraTestClusterBuilderCustomizer(TestCassandra cassandra) {
			return new CassandraClusterBuilderCustomizer(cassandra.getSettings());
		}

	}

	/**
	 * Configuration that overrides {@link Cluster} configuration via {@link ClusterBuilderCustomizer} bean using
	 * {@link Cassandra} bean properties.
	 */
	@Configuration
	@ConditionalOnMissingBean(type = "com.github.nosan.embedded.cassandra.test.TestCassandra")
	@ConditionalOnSingleCandidate(Cassandra.class)
	static class CassandraClusterConfiguration {

		@Bean
		@Order(0)
		public ClusterBuilderCustomizer cassandraClusterBuilderCustomizer(Cassandra cassandra) {
			return new CassandraClusterBuilderCustomizer(cassandra.getSettings());
		}

	}

	static class CassandraClusterBuilderCustomizer implements ClusterBuilderCustomizer {

		private final Settings settings;

		CassandraClusterBuilderCustomizer(Settings settings) {
			this.settings = settings;
		}

		@Override
		public void customize(Cluster.Builder builder) {
			Settings settings = this.settings;
			InetAddress address = settings.address().orElse(null);
			Integer port = settings.portOrSslPort().orElse(null);
			if (address != null && port != null) {
				builder.addContactPoints(address).withPort(port);
			}
		}

	}

}
