//
// JODConverter - Java OpenDocument Converter
// Copyright 2004-2012 Mirko Nasato and contributors
//
// JODConverter is Open Source software, you can redistribute it and/or
// modify it under either (at your option) of the following licenses
//
// 1. The GNU Lesser General Public License v3 (or later)
//    -> http://www.gnu.org/licenses/lgpl-3.0.txt
// 2. The Apache License, Version 2.0
//    -> http://www.apache.org/licenses/LICENSE-2.0.txt
//
package com.luues.openoffice.core.office;

import com.luues.openoffice.config.ApplicationData;
import com.luues.openoffice.core.util.PlatformUtils;
import com.sun.star.beans.PropertyValue;
import com.sun.star.uno.UnoRuntime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import java.io.*;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.Enumeration;
import java.util.Map;
import java.util.jar.JarEntry;

@Component
public class OfficeUtils {
    private static boolean flag = true;
    private static boolean flags = true;
    private static final String jar_name = "commons-openoffice.jar";
    private static final String bacapp_name = "application-bac.properties";
    private static final Logger logger = LoggerFactory.getLogger(OfficeUtils.class);
    public static final String SERVICE_DESKTOP = "com.sun.star.frame.Desktop";
    @Autowired
    private ApplicationData applicationData;

    public static <T> T cast(Class<T> type, Object object) {
        return (T) UnoRuntime.queryInterface(type, object);
    }

    public static PropertyValue property(String name, Object value) {
        PropertyValue propertyValue = new PropertyValue();
        propertyValue.Name = name;
        propertyValue.Value = value;
        return propertyValue;
    }

    @SuppressWarnings("unchecked")
    public static PropertyValue[] toUnoProperties(Map<String, ?> properties) {
        PropertyValue[] propertyValues = new PropertyValue[properties.size()];
        int i = 0;
        for (Map.Entry<String, ?> entry : properties.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Map) {
                Map<String, Object> subProperties = (Map<String, Object>) value;
                value = toUnoProperties(subProperties);
            }
            propertyValues[i++] = property((String) entry.getKey(), value);
        }
        return propertyValues;
    }

    public static String toUrl(File file) {
        String path = file.toURI().getRawPath();
        String url = path.startsWith("//") ? "file:" + path : "file://" + path;
        return url.endsWith("/") ? url.substring(0, url.length() - 1) : url;
    }

    public File getDefaultOfficeHome() {
        if(null != applicationData.getOfficeHome() && !"null".equals(applicationData.getOfficeHome())){
            return new File(applicationData.getOfficeHome());
        }
        if (PlatformUtils.isWindows()) {
            // %ProgramFiles(x86)% on 64-bit machines; %ProgramFiles% on 32-bit ones
            String programFiles = System.getenv("ProgramFiles(x86)");
            if (programFiles == null) {
                programFiles = System.getenv("ProgramFiles");
            }
            return findOfficeHome(
                    programFiles + File.separator + "OpenOffice 4",
                    programFiles + File.separator + "LibreOffice 4",
                    ""
            );
        } else if (PlatformUtils.isMac()) {
            return findOfficeHome(
                    "/Applications/OpenOffice.org.app/Contents",
                    "/Applications/OpenOffice.app/Contents",
                    "/Applications/LibreOffice.app/Contents"
            );
        } else {
            // Linux or other *nix variants
            return findOfficeHome(
                    "/opt/openoffice.org3",
                    "/opt/openoffice",
                    "/opt/libreoffice",
                    "/opt/openoffice4",
                    "/usr/lib/openoffice",
                    "/usr/lib/libreoffice"
            );
        }
    }

    private static File findOfficeHome(String... knownPaths) {
        for (String path : knownPaths) {
            File home = new File(path);
            if (getOfficeExecutable(home).isFile()) {
                return home;
            }
        }
        logger.error("启动openoffice失败, 请先安装openoffice, 如已安装，请添加配置:{}", "office.home=");
        return null;
    }

    public static File getOfficeExecutable(File officeHome) {
        if (PlatformUtils.isMac()) {
            return new File(officeHome, "MacOS/soffice.bin");
        } else {
            return new File(officeHome, "program/soffice.bin");
        }
    }

    /*public static String getHomePath() {
        return OfficeUtils.class.getClassLoader().getResource("").getPath();*/
        /*String userDir = System.getenv("KKFILEVIEW_BIN_FOLDER");
        if (userDir == null) {
            userDir = System.getProperty("user.dir");
        }
        if (userDir.endsWith("bin")) {
            userDir = userDir.substring(0, userDir.length() - 4);
        } else {
            userDir = OfficeUtils.class.getResource("/office").getPath();
        }*/

        /*try {
            if (userDir.contains(".jar!")) {
                String tmpdir = System.getProperty("java.io.tmpdir");
                tmpdir = tmpdir.lastIndexOf(File.separator) == tmpdir.length() - 1 ? tmpdir + "cus_office" : tmpdir + File.separator + "cus_office";
                System.setProperty("java.library.path", tmpdir + File.separator + "office");
                String tmp_path = tmpdir;
                int count = (userDir.length() - userDir.replace(".jar", "").length()) / ".jar".length();
                if(count == 1){
                    File file = new File(tmp_path);
                    synchronized (String.valueOf(flag)) {
                        if (flag) {
                            flag = false;
                            logger.warn("Temporary file generation... , Route is {} , Please wait a moment", tmp_path);
                            if (!file.exists()) {
                                file.mkdirs();
                            }
                            file = new File(tmp_path, "application.properties");
                            if (file.exists()) {
                                return tmp_path + File.separator + "office";
                            }
                            if (PlatformUtils.isWindows()) {
                                String jarPath = OfficeUtils.class.getProtectionDomain().getCodeSource().getLocation().getFile();
                                jarPath = URLDecoder.decode(jarPath, "UTF-8");
                                JarFile files = new JarFile(jarPath);
                                Enumeration<JarEntry> entries = files.entries();
                                createTmp(entries, tmp_path, jarPath);
                            }
                        }
                    }
                    return tmp_path + File.separator + "office";
                }else{
                    File file = new File(tmp_path);
                    synchronized (String.valueOf(flag)) {
                        if (flag) {
                            flag = false;
                            logger.warn("Temporary file generation... , Route is {} , Please wait a moment", tmp_path);
                            if (!file.exists()) {
                                file.mkdirs();
                            }
                            file = new File(tmp_path, "application.properties");
                            if (file.exists()) {
                                return tmp_path + File.separator + "office";
                            }
                            if (PlatformUtils.isWindows()) {
                                URL url = new URL("jar:" + userDir.replace("!/office", ""));
                                Files.copy(url.openStream(), new File(tmp_path + File.separator + jar_name).toPath(), StandardCopyOption.REPLACE_EXISTING);
                                String jarPath = tmp_path + File.separator + jar_name;
                                jarPath = URLDecoder.decode(jarPath, "UTF-8");
                                JarFile files = new JarFile(jarPath);
                                Enumeration<JarEntry> entries = files.entries();
                                createTmp(entries, tmp_path, jarPath);
                                new File(jarPath).delete();
                            }
                        }
                    }
                    return tmp_path + File.separator + "office";
                }
            } else {
                return OfficeUtils.class.getResource("/office").getPath();
            }
        } catch (IOException e) {
        }
        return null;*/
    //}

    /*public static String getCustomizedConfigPath() {
        String homePath = OfficeUtils.getHomePath();
        return homePath;*/
        /*homePath = homePath.contains(File.separator + "office") ? homePath.replace(File.separator + "office", "") : homePath;
        synchronized (String.valueOf(flags)){
            if(flags){
                flags = false;
                URL url = OfficeUtils.class.getClassLoader().getResource("application.properties");
                if (null != url) {
                    try {
                        FileWriter fileWriter =new FileWriter(homePath + File.separator + bacapp_name);
                        BufferedWriter bufferedWriter =new BufferedWriter(fileWriter);
                        InputStreamReader inputStreamReader = new InputStreamReader(url.openStream());
                        BufferedReader br = new BufferedReader(inputStreamReader); // 建立一个对象，它把文件内容转成计算机能读懂的语言
                        String line;
                        while ((line = br.readLine()) != null) {
                            bufferedWriter.write(line + "\n");
                        }
                        bufferedWriter.close();
                        mergeFiles(homePath + File.separator + "application_.properties", new String[]{homePath + File.separator + bacapp_name, homePath + File.separator + "application.properties"});
                        new File(homePath + File.separator + bacapp_name).delete();
                        new File(homePath + File.separator + "application.properties").delete();
                        new File(homePath + File.separator + "application_.properties").renameTo(new File(homePath + File.separator + "application.properties"));
                        duplicateRemoval(homePath + File.separator + "application.properties");
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
                return homePath + File.separator + "application.properties";
            }
            return homePath + File.separator + "application.properties";
        }*/
    //}

    private static void createTmp(Enumeration<JarEntry> entries, String tmp_path, String jarPath) throws IOException {
        while (entries.hasMoreElements()) {
            JarEntry jarEntry = entries.nextElement();
            if (jarEntry.getName().startsWith("office/")) {
                if (jarEntry.isDirectory()) {
                    File file_ = new File(tmp_path + File.separator + jarEntry.getName());
                    if (!file_.exists()) {
                        file_.mkdirs();
                    }
                } else {
                    File file2 = new File(tmp_path + File.separator + "office" + File.separator + jarEntry.getName().split("office/")[1]);
                    URL url1 = new URL("jar:file:" + jarPath + "!/" + "office" + "/" + jarEntry.getName().split("office/")[1]);
                    Files.copy(url1.openStream(), file2.toPath(), StandardCopyOption.REPLACE_EXISTING);
                }
            }
            if (jarEntry.getName().equals("application.properties")) {
                URL url1 = new URL("jar:file:" + jarPath + "!/" + jarEntry.getName());
                File file2 = new File(tmp_path + File.separator + jarEntry.getName());
                Files.copy(url1.openStream(), file2.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
        }
    }

    private static void duplicateRemoval(String file) throws IOException {
        BufferedReader bufferedReader = new BufferedReader(new FileReader(file));
        StringBuffer stringBuffer = new StringBuffer();
        String str;
        while ((str = bufferedReader.readLine()) != null) {
            stringBuffer.append(str + "\n");
        }
        bufferedReader.close();
        StringBuffer add = new StringBuffer("\n");
        StringBuffer newStr = new StringBuffer();
        String content = stringBuffer.toString();
        for(String string : content.split("\n")){
            if(string.length() == 0){
                newStr.append("\n");
                continue;
            }
            int count = (content.length() - content.replace(string, "").length()) / string.length();
            if(count > 1){
                boolean f = true;
                for(String s : add.toString().split("\n")){
                    if(s.equals(string)){
                       f = false;
                    }
                }
                if(f){
                    add.append(string + "\n");
                    newStr.append(string + "\n");
                }
            }else{
                newStr.append(string + "\n");
            }
        }
        File file1 = new File(new File(file).getParent() + File.separator + "application-bac.properties");
        BufferedWriter out = new BufferedWriter(new FileWriter(file1.getPath()));
        out.write(newStr.toString());
        out.close();
        new File(file).delete();
        file1.renameTo(new File(file1.getParent() + File.separator + "application.properties"));
    }

    private static void mergeFiles(String outFile, String[] files) {
        FileChannel outChannel = null;
        try {
            outChannel = new FileOutputStream(outFile).getChannel();
            for(String file : files){
                FileChannel fc = new FileInputStream(file).getChannel();
                ByteBuffer bb = ByteBuffer.allocate(1024);
                while(fc.read(bb) != -1){
                    bb.flip();
                    outChannel.write(bb);
                    bb.clear();
                }
                fc.close();
            }
        } catch (IOException ioe) {
            ioe.printStackTrace();
        } finally {
            try {if (outChannel != null) {outChannel.close();}} catch (IOException ignore) {}
        }
    }

}
