package com.github.lontime.base.workflow.provider;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.github.lontime.base.api.workflow.ActionType;
import com.github.lontime.base.api.workflow.ReactiveInterface;
import com.github.lontime.base.api.workflow.StatefulInterface;
import com.github.lontime.base.commonj.utils.CollectionHelper;
import com.github.lontime.base.commonj.utils.LoggerHelper;
import com.github.lontime.base.commonj.utils.StringHelper;
import com.github.lontime.base.serial.codec.MsgCodec;
import com.github.lontime.base.serial.mq.Message;
import com.github.lontime.base.workflow.configuration.OptionResolver;
import com.github.lontime.base.workflow.configuration.Options;
import com.github.lontime.extredisson.RemoteInstance;
import com.github.lontime.extredisson.remote.common.ReplyOutput;

/**
 * AbstractProvider.
 * @author lontime
 * @since 1.0
 */
public abstract class AbstractProvider implements Provider {

    @Override
    public <REQ, RES> RES state(String schemaName, String methodName, REQ params, MsgCodec<REQ, RES> codec) {
        final Options.Schema schema = OptionResolver.getInstance().getSchema(schemaName);
        if (schema == null) {
            LoggerHelper.warn(getClass(), "cannot find {0}", schemaName);
            return null;
        }
        final Message message = Message.builder().serviceName(schema.getServiceName())
                .methodName(methodName).body(params, codec.getIndata()).build();
        final ReplyOutput output = RemoteInstance.get().call(schema.getProxy(), message);
        if (output == null) {
            return null;
        }
        return output.resolveResponse(codec.getOutdata());
    }

    @Override
    public <REQ, RES> RES restart(String schemaName, String stepName, REQ params, MsgCodec<REQ, RES> codec) {
        return pushInternal(schemaName, stepName, params, codec, ActionType.RESTART);
    }

    @Override
    public <REQ, RES> RES start(String schemaName, String stepName, REQ params, MsgCodec<REQ, RES> codec) {
        return pushInternal(schemaName, stepName, params, codec, ActionType.START);
    }

    @Override
    public <REQ, RES> RES end(String schemaName, String stepName, REQ params, MsgCodec<REQ, RES> codec) {
        return pushInternal(schemaName, stepName, params, codec, ActionType.END);
    }

    @Override
    public <REQ, RES> RES destroy(String schemaName, String stepName, REQ params, MsgCodec<REQ, RES> codec) {
        return pushInternal(schemaName, stepName, params, codec, ActionType.DESTROY);
    }

    @Override
    public <REQ, RES> RES push(String schemaName, String stepName, REQ params, MsgCodec<REQ, RES> codec) {
        return pushInternal(schemaName, stepName, params, codec, ActionType.NORMAL);
    }

    private <REQ, RES> RES pushInternal(String schemaName, String stepName, REQ params, MsgCodec<REQ, RES> codec, ActionType actionType) {
        final Options.Schema schema = OptionResolver.getInstance().getSchema(schemaName);
        if (schema == null) {
            LoggerHelper.warn(getClass(), "cannot find {0}", schemaName);
            return null;
        }
        final Options.Step step = OptionResolver.getInstance().getStep(schema, stepName);
        if (step == null) {
            LoggerHelper.warn(getClass(), "cannot find {0}/{1}", schemaName, stepName);
            return null;
        }
        if (params instanceof ReactiveInterface) {
            if (!attachParams(schema.getName(), step, (ReactiveInterface) params, actionType)) {
                return null;
            }
        }
        final Message.Builder messageBuilder = Message.builder().serviceName(step.getServiceName())
                .methodName(step.getMethodName()).body(params, codec.getIndata());
        final ReplyOutput output = RemoteInstance.get().call(step.getProxy(), messageBuilder.build());
        if (output == null) {
            return null;
        }
        final RES response = output.resolveResponse(codec.getOutdata());
        if (response instanceof StatefulInterface) {
            if (params instanceof ReactiveInterface) {
                attachResponse(step, ((ReactiveInterface) params).getBid(), (StatefulInterface) response);
            } else {
                attachResponse(step, null, (StatefulInterface) response);
            }
        }
        if (StringHelper.hasText(step.getNextStep())) {
            final RES nextResponse = pushInternal(schemaName, step.getNextStep(), params, codec, actionType);
            if (nextResponse != null) {
                return nextResponse;
            }
        }
        return response;
    }

    private boolean attachParams(String schemaName, Options.Step option, ReactiveInterface params, ActionType actionType) {
        final String bid = params.getBid();
        if (StringHelper.isEmpty(bid)) {
            LoggerHelper.warn(getClass(), "business id is not null");
            return false;
        }
        final String nextState = option.getNextState();
        if (actionType != ActionType.DESTROY && actionType != ActionType.RESTART) {
            final List<String> states = option.getStates();
            final String state = params.getState();
            if (Objects.isNull(state)) {
                if (CollectionHelper.isNotEmpty(states) && states.size() == 1) {
                    params.setState(states.get(0));
                } else {
                    throw new IllegalStateException("state is null");
                }
            } else if (CollectionHelper.isNotEmpty(states)) {
                if (states.stream().noneMatch(s -> s.equalsIgnoreCase(state))) {
                    throw new IllegalStateException("Illegal state " + state + " in "
                            + states.stream().collect(Collectors.joining(",")));
                }
            }
        }
        params.setSchema(schemaName);
        params.setActionType(actionType);
        params.setNextState(nextState);
        params.setAction(option.getAction());
        return true;
    }

    private void attachResponse(Options.Step step, String bid, StatefulInterface response) {
        if (response.getBid() == null && bid != null) {
            response.setBid(bid);
        }
        if (response.getState() == null) {
            response.setState(step.getNextState());
        }
    }

}
