package com.github.lontime.base.workflow.impl;

import java.util.Objects;

import com.github.lontime.base.api.workflow.ActionType;
import com.github.lontime.base.api.workflow.ReactiveInterface;
import com.github.lontime.base.api.workflow.StatefulInterface;
import com.github.lontime.base.serial.MsgpackSerial;
import com.github.lontime.base.workflow.interfaces.WfPersistentInterface;
import com.github.lontime.extredisson.consumer.AbstractRemoteService;
import com.github.lontime.extredisson.remote.common.RemoteContext;

/**
 * AbstractWorkflowService.
 * @author lontime
 * @since 1.0
 * @param <REQ> request
 * @param <RES> response
 */
public abstract class AbstractWorkflowService<REQ extends ReactiveInterface, RES extends StatefulInterface>
        extends AbstractRemoteService<REQ, RES>  {

    private WfPersistentInterface<REQ> persistentInterface;

    public AbstractWorkflowService(WfPersistentInterface<REQ> wfPersistentInterface) {
        this.persistentInterface = wfPersistentInterface;
    }

    @Override
    public RES beforeRoute(RemoteContext context, REQ inbound) {
        if (inbound.getActionType() == ActionType.NVL) {
            return null;
        }
        final String state = inbound.getState();
        if (inbound.getActionType() == ActionType.NORMAL || inbound.getActionType() == ActionType.END) {
            final String persistentState = queryState(inbound);
            if (!Objects.equals(state, persistentState)) {
                return errorRoute(context, inbound, new IllegalArgumentException("IllegalArgument for route"));
            }
        }
        return null;
    }

    @Override
    public RES afterRoute(RemoteContext context, REQ inbound, RES outbound) {
        if (inbound.getActionType() == ActionType.NVL) {
            return outbound;
        }
        final byte[] requestBytes = MsgpackSerial.INSTANCE.toByteArrayUnchecked(inbound, codec().getIndata());
        final byte[] responseBytes = MsgpackSerial.INSTANCE.toByteArrayUnchecked(outbound, codec().getOutdata());
        if (inbound.getActionType() == ActionType.START) {
            persistentInterface.insertStateAndAuditTrail(inbound, requestBytes, responseBytes);
        } else if (inbound.getActionType() != ActionType.RESTART) {
            persistentInterface.updateStateAndAuditTrail(inbound, requestBytes, responseBytes);
        }
        return outbound;
    }

    /**
     * state.
     * @return state
     */
    public String queryState(REQ inbound) {
        return persistentInterface.queryState(inbound);
    }

}
