package com.github.lontime.base.workflow.impl;

import java.sql.Connection;
import java.sql.SQLException;

import com.github.lontime.base.api.workflow.ReactiveInterface;
import com.github.lontime.base.commonj.utils.LoggerHelper;
import com.github.lontime.base.workflow.interfaces.WfRdbmsInterface;
import com.github.lontime.extdatasource.DbInstance;
import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.handlers.ScalarHandler;

/**
 * WfRdbmsInterface
 * @author lontime
 * @since 1.0
 * @param <REQ> request
 */
public abstract class AbstractWfRdbmsImpl<REQ extends ReactiveInterface> implements WfRdbmsInterface<REQ> {

    private String dsName;

    public AbstractWfRdbmsImpl(String dsName) {
        this.dsName = dsName;
    }

    protected int insertState(Connection connection, REQ request) throws SQLException {
        final QueryRunner qr = new QueryRunner();
        //insert into tb_state(bid, schema_name, state, created_time, updated_time) values(?, ?, NOW(), NOW())
        final String sql = getInsertStateSql();
        return qr.update(connection, sql, request.getBid(), request.getSchema(), request.getNextState());
    }

    @Override
    public int insertState(REQ request) {
        return DbInstance.get().handle(dsName, connection -> {
            return insertState(connection, request);
        }).orElse(-1);
    }

    protected int updateState(Connection connection, REQ request) throws SQLException {
        //update tb_state set state = ?, updated_time = NOW() where bid = ? and schema_name = ?
        final String sql = getUpdateStateSql();
        final QueryRunner qr = new QueryRunner();
        return qr.update(connection, sql, request.getNextState(), request.getBid(), request.getSchema());
    }

    @Override
    public int updateState(REQ request) {
        return DbInstance.get().handle(dsName, connection -> {
            return updateState(connection, request);
        }).orElse(-1);
    }

    @Override
    public String queryState(REQ request) {
        //select state from tb_state where bid = ? and schema_name = ?
        final String sql = getQueryStateSql();
        return DbInstance.get().handle(dsName, connection -> {
            final QueryRunner qr = new QueryRunner();
            return qr.<String>query(connection, sql, new ScalarHandler<>(),
                    request.getBid(), request.getSchema());
        }).orElse(null);
    }

    protected int insertAuditTrail(Connection connection, REQ request, byte[] params, byte[] response) throws SQLException {
        //bid, schema_name, action_type, action, prev_state, state, params, response
        final String sql = getInsertAuditTrailSql();
        final QueryRunner qr = new QueryRunner();
        final Number newId = qr.insert(connection, sql, new ScalarHandler<>(), request.getBid(), request.getSchema(),
                request.getActionType().name(), request.getAction(),
                request.getState(), request.getNextState(), params, response);
        LoggerHelper.debug(getClass(), "insertAuditTrail new id {0}", newId.toString());
        return newId.intValue() > 0 ? 1 : 0;
    }

    @Override
    public int insertAuditTrail(REQ request, byte[] params, byte[] response) {
        return DbInstance.get().handle(dsName, connection -> {
            return insertAuditTrail(connection, request, params, response);
        }).orElse(-1);
    }

    @Override
    public int insertStateAndAuditTrail(REQ request, byte[] params, byte[] response) {
        return DbInstance.get().tx(dsName, connection -> {
            int result = 0;
            result += insertState(connection, request);
            result += insertAuditTrail(connection, request, params, response);
            return result;
        }).orElse(-1);
    }

    @Override
    public int updateStateAndAuditTrail(REQ request, byte[] params, byte[] response) {
        return DbInstance.get().tx(dsName, connection -> {
            int result = 0;
            result += updateState(connection, request);
            result += insertAuditTrail(connection, request, params, response);
            return result;
        }).orElse(-1);
    }
}
