package com.github.lontime.base.workflow.configuration;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;

import com.github.lontime.base.commonj.utils.SupplierHelper;
import com.github.lontime.extconfig.ConfigHelper;
import com.github.lontime.shaded.com.google.common.base.Splitter;
import com.github.lontime.shaded.io.helidon.config.Config;

/**
 * Workflow Option Resolver.
 *
 * @author lontime
 * @since 1.0
 */
public class OptionResolver {

    private static final String NAME = "workflow";

    //------------------singleton------------------
    /**
     * supplier.
     */
    private static Supplier<OptionResolver> supplier = SupplierHelper.memoize(OptionResolver::new);

    public static OptionResolver getInstance() {
        return supplier.get();
    }
    //------------------singleton------------------

    private final Options options;


    public OptionResolver() {
        this.options = ConfigHelper.resolve(NAME).as(new Mapper()).orElse(new Options());
    }

    public Options.Schema getSchema(String name) {
        return options.getSchemas().stream().filter(s -> s.getName().equals(name))
                .findFirst().orElse(null);
    }

    public Options.Step getStep(Options.Schema schema, String name) {
        return schema.getSteps().stream().filter(s -> s.getName().equals(name))
                .findFirst().orElse(null);
    }

    private static class Mapper implements Function<Config, Options> {

        @Override
        public Options apply(Config config) {
            final Options internal = new Options();
            config.get("schemas").asList(new SchemaMapper()).ifPresent(internal::setSchemas);
            return internal;
        }
    }

    private static class SchemaMapper implements Function<Config, Options.Schema> {

        @Override
        public Options.Schema apply(Config config) {
            final String name = config.get("name").asString().orElse(config.name());
            final String serviceName = config.get("serviceName").asString().orElse(name);

            final Options.Schema option = new Options.Schema();
            option.setName(name);
            option.setServiceName(serviceName);
            config.get("proxy").asString().ifPresent(option::setProxy);
            config.get("version").asInt().ifPresent(option::setVersion);
            config.get("stepPath").asString().map(s -> loadStep(s, option.getProxy(), serviceName)).ifPresent(option::setSteps);
            config.get("steps").asList(new StepMapper(option.getProxy(), serviceName)).ifPresent(option::setSteps);
            return option;
        }

        private List<Options.Step> loadStep(String name, String proxy, String serviceName) {
            return ConfigHelper.resolve(name).asList(new StepMapper(proxy, serviceName)).orElse(null);
        }
    }

    private static class StepMapper implements Function<Config, Options.Step> {

        private String proxy;
        private String serviceName;

        public StepMapper(String proxy, String serviceName) {
            this.proxy = proxy;
            this.serviceName = serviceName;
        }

        @Override
        public Options.Step apply(Config config) {
            final String name = config.get("name").asString().orElse(config.name());
            final String serviceNameInner = config.get("serviceName").asString().orElse(serviceName);
            final String methodName = config.get("methodName").asString().orElse(name);
            final String actionName = config.get("action").asString().orElse(name);

            final Options.Step option = new Options.Step();
            option.setName(name);
            option.setAction(actionName);
            option.setMethodName(methodName);
            option.setServiceName(serviceNameInner);
            final Optional<String> proxyOptional = config.get("proxy").asString().asOptional();
            if (proxyOptional.isPresent()) {
                option.setProxy(proxyOptional.get());
            } else {
                option.setProxy(proxy);
            }
            config.get("state").asString().map(Collections::singletonList).ifPresent(option::setStates);
            config.get("states").asString().map(s -> Splitter.on(",").omitEmptyStrings().trimResults().splitToList(s))
                    .ifPresent(option::setStates);
            config.get("nextState").asString().ifPresent(option::setNextState);
            config.get("nextStep").asString().ifPresent(option::setNextStep);
            return option;
        }
    }

}