package com.github.lontime.base.errors;

import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.github.lontime.base.errors.spi.ErrorMessageDetector;

/**
 * @author lontime
 * @since 1.0
 */
public class ErrorMessageDetectorRegistry {

    private static Map<String, ErrorMessageDetector> map = new ConcurrentHashMap<>();

    public static ErrorMessageDetector getOrSet(String lang, Function<String, ErrorMessageDetector> detector) {
        return map.computeIfAbsent(lang, detector);
    }

    public static ErrorMessageDetector getOrSet(Function<String, ErrorMessageDetector> detector) {
        return getOrSet(getDefaultLang(), detector);
    }

    public static ErrorMessageDetector get(String lang) {
        if (Locale.CHINESE.getLanguage().equals(lang)) {
            return getOrSet(lang, key -> new ChineseErrorMessageDetector());
        }
        return map.get(lang);
    }

    private static String getDefaultLang() {
        final String lang = System.getenv("ERRORS_MESSAGE_LANG");
        if (lang != null) {
            return lang;
        }
        return System.getProperty("errors.message.lang", Locale.getDefault().getLanguage());
    }

    public static ErrorMessageDetector get() {
        return get(getDefaultLang());
    }

    private static void loadDefault() {
        get();
    }

    public static List<ErrorMessageDetector> getAll() {
        loadDefault();
        return map.values().stream().sorted(Comparator.comparingInt(ErrorMessageDetector::sortNum))
                .collect(Collectors.toList());
    }

}