package com.github.lontime.base.errors;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.stream.Collectors;

import am.ik.yavi.core.ConstraintViolation;
import am.ik.yavi.core.ConstraintViolations;

/**
 * error exception.
 * @author lontime
 * @since 1.0.0
 */
public class ErrorException extends RuntimeException {
    private final String code;
    private final Integer statusCode;

    public ErrorException(ErrorCodeEnum codeEnum, String message) {
        super(message);
        this.code = codeEnum.getCode();
        this.statusCode = ErrorTypes.detectErrorType(codeEnum).getStatusCode();
    }

    public ErrorException(String code, String message) {
        super(message);
        this.code = code;
        this.statusCode = ErrorTypes.detectErrorType(code).getStatusCode();
    }

    public ErrorException(String code, String message, Throwable cause) {
        super(message, cause);
        this.code = code;
        this.statusCode = ErrorTypes.detectErrorType(code).getStatusCode();
    }

    public String getCode() {
        return code;
    }

    public Integer getStatusCode() {
        return statusCode;
    }

    /**
     * ErrorException.
     * @param errorCodeEnum errcode
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException from(ErrorCodeEnum errorCodeEnum) {
        return from(errorCodeEnum.getCode());
    }

    /**
     * ErrorException.
     * @param code errcode
     * @param cause 原因
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException from(String code, Throwable cause) {
        return new ErrorException(code, ErrorTypes.detectErrorType(code).getMessage(), cause);
    }

    /**
     * ErrorException.
     * @param errorCodeEnum errcode
     * @param message 消息
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException from(ErrorCodeEnum errorCodeEnum, String message) {
        return new ErrorException(errorCodeEnum.getCode(), message);
    }

    /**
     * ErrorException.
     * @param errorCodeEnum errcode
     * @param cause 原因
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException from(ErrorCodeEnum errorCodeEnum, Throwable cause) {
        return from(errorCodeEnum.getCode(), cause);
    }

    /**
     * ErrorException.
     * @param errorCodeEnum errcode
     * @param message 消息
     * @param cause 原因
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException from(ErrorCodeEnum errorCodeEnum, String message, Throwable cause) {
        return new ErrorException(errorCodeEnum.getCode(), message, cause);
    }

    /**
     * ErrorException.
     * @param code errcode
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException from(String code) {
        return new ErrorException(code, ErrorTypes.detectErrorType(code).getMessage());
    }

    /**
     * ErrorException.
     * @param violations violations
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException invalid(List<ConstraintViolation> violations) {
        final String message = violations.stream().map(s -> s.message())
                .collect(Collectors.joining(";"));
        return new ErrorException(ErrorCodeEnum.ERROR_PARAM, message);
    }

    /**
     * ErrorException.
     * @param violations violations
     * @return ErrorException
     * @author lontime
     * @since 1.0.0
     */
    public static ErrorException invalid(ConstraintViolations violations) {
        final String message = violations.stream().map(s -> s.message())
                .collect(Collectors.joining(";"));
        return new ErrorException(ErrorCodeEnum.ERROR_PARAM, message);
    }

    /**
     * getFullStackTrace.
     * @param throwable throwable
     * @return string
     */
    public static String getFullStackTrace(Throwable throwable) {
        final Charset charset = StandardCharsets.UTF_8;
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        PrintStream ps = null;
        try {
            ps = new PrintStream(baos, true, charset.name());
            throwable.printStackTrace(ps);
            return baos.toString(charset.name());
        } catch (IOException ex) {
            //ignore
        } finally {
            if (ps != null) {
                ps.close();
            }
            try {
                baos.close();
            } catch (IOException ios) {
                //ignore
            }
        }
        return "";
    }
}
