package com.github.kuliginstepan.outbox.core;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.reactive.TransactionSynchronization;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

@Slf4j
@RequiredArgsConstructor
class ReactiveOutboxTransactionSynchronization implements TransactionSynchronization {

    private final ReactiveOutboxRepository repository;
    private final OutboxEntity entity;
    private final MethodInvoker invoker;

    @Override
    public Mono<Void> afterCommit() {
        log.debug("Invoking outbox method: {}, args: {}", invoker.getMethod(), invoker.getArgs());
        Mono.defer(() -> ((Mono<?>) invoker.invoke()))
            .doOnSuccess(o -> log.debug("Outbox method {} successfully invoked", invoker.getMethod()))
            .then(Mono.defer(() -> repository.markCompleted(entity)))
            .doOnSuccess(o -> log.debug("Outbox method {} marked as completed", invoker.getMethod()))
            .subscribeOn(Schedulers.boundedElastic())
            .subscribe(
                o -> log.debug("Outbox method {} marked as completed", invoker.getMethod()),
                e -> log.error("Outbox method {} invoked with error", invoker.getMethod(), e)
            );
        return Mono.empty();
    }
}
