package com.github.kuliginstepan.outbox.core;

import java.util.function.Supplier;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.transaction.NoTransactionException;
import org.springframework.transaction.reactive.TransactionSynchronizationManager;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;

@Slf4j
@Aspect
@RequiredArgsConstructor
public class ReactiveOutboxAspect {

    private final OutboxEntityFactory entityFactory;
    private final ReactiveOutboxRepository repository;
    private final Scheduler scheduler;

    @Pointcut("@annotation(Outbox) && execution(reactor.core.publisher.Mono *(..))")
    public void outboxReactiveMethod() {}

    @Around("outboxReactiveMethod()")
    public Object outboxAroundAspect(ProceedingJoinPoint pjp) {
        MethodSignature signature = (MethodSignature) pjp.getSignature();
        OutboxEntity entity = entityFactory
            .create(pjp.getArgs(), OutboxMethodIdentifier.ofMethod(signature.getMethod()));
        MethodInvoker methodInvoker = new MethodInvoker(signature.getMethod(), pjp.getTarget(), pjp.getArgs());

        Supplier<Mono<Void>> noTransactionHandler = () -> {
            log.warn("Execute outbox method {} without active transaction. It can lead to unexpected behaviour",
                signature.getMethod());
            repository.save(entity)
                .doOnSuccess(
                    it -> log.debug("Invoking outbox method: {}, args: {}", signature.getMethod(), pjp.getArgs()))
                .then(Mono.defer(() -> (Mono<?>) methodInvoker.invoke()))
                .then(Mono.defer(() -> repository.markCompleted(entity)))
                .subscribeOn(scheduler)
                .subscribe(
                    o -> log.debug("Outbox method {} marked as completed", signature.getMethod()),
                    e -> log.error("Outbox method {} invoked with error", signature.getMethod(), e)
                );
            return Mono.empty();
        };

        return TransactionSynchronizationManager.forCurrentTransaction()
            .filter(TransactionSynchronizationManager::isActualTransactionActive)
            .filter(TransactionSynchronizationManager::isSynchronizationActive)
            .flatMap(manager ->
                repository.save(entity)
                    .thenReturn(manager)
                    .handle((it, sink) -> {
                        manager.registerSynchronization(
                            new ReactiveOutboxTransactionSynchronization(repository, entity, methodInvoker)
                        );
                        sink.next(manager);
                    })
            )
            .switchIfEmpty(Mono.defer(noTransactionHandler))
            .onErrorResume(NoTransactionException.class, t -> noTransactionHandler.get())
            .then();
    }
}
