package com.github.kuliginstepan.outbox.core;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

public class OutboxMethodIdentifier {

    private static final Pattern regex = Pattern.compile("\\[(.*)]\\{(.*)}\\((.*)\\)");
    private final String value;

    public String getValue() {
        return value;
    }

    public OutboxMethodIdentifier(String value) {this.value = value;}

    public static OutboxMethodIdentifier of(String value) {
        return new OutboxMethodIdentifier(value);
    }

    public static OutboxMethodIdentifier ofMethod(Method method) {
        String typeName = ClassUtils.getUserClass(method.getDeclaringClass()).getName();
        String methodName = method.getName();
        String parameterTypes = Arrays.stream(method.getParameterTypes())
            .map(Class::getName)
            .collect(Collectors.joining(","));
        return OutboxMethodIdentifier.of(String.format("[%s]{%s}(%s)", typeName, methodName, parameterTypes));
    }

    public Class<?> getOutboxClass() {
        return ClassUtils.resolveClassName(getMatchedGroup(1), OutboxMethodIdentifier.class.getClassLoader());
    }

    public String getMethodName() {
        return getMatchedGroup(2);
    }

    public Class<?>[] getParameterTypes() {
        return Arrays.stream(StringUtils.delimitedListToStringArray(getMatchedGroup(3), ","))
            .map(paramType -> ClassUtils.resolveClassName(paramType, OutboxMethodIdentifier.class.getClassLoader()))
            .toArray(Class[]::new);
    }

    private String getMatchedGroup(int group) {
        Matcher matcher = regex.matcher(value);
        if (matcher.find()) {
            return matcher.group(group);
        } else {
            return null;
        }
    }
}
