package com.github.kuliginstepan.outbox.core;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.task.TaskExecutor;
import org.springframework.transaction.support.TransactionSynchronizationManager;

@Slf4j
@Aspect
@RequiredArgsConstructor
public class OutboxAspect {

    private final OutboxEntityFactory entityFactory;
    private final OutboxRepository repository;
    private final TaskExecutor executor;

    @Pointcut("@annotation(Outbox) && execution(void *(..))")
    public void outboxBlockingMethod() {}

    @Around("outboxBlockingMethod()")
    public Object outboxAroundAspect(ProceedingJoinPoint pjp) {
        MethodSignature signature = (MethodSignature) pjp.getSignature();
        OutboxEntity entity = entityFactory
            .create(pjp.getArgs(), OutboxMethodIdentifier.ofMethod(signature.getMethod()));
        MethodInvoker methodInvoker = new MethodInvoker(signature.getMethod(), pjp.getTarget(), pjp.getArgs());
        if (TransactionSynchronizationManager.isSynchronizationActive() &&
            TransactionSynchronizationManager.isActualTransactionActive()) {
            repository.save(entity);
            TransactionSynchronizationManager.registerSynchronization(
                new OutboxTransactionSynchronization(repository, entity, executor, methodInvoker)
            );
        } else {
            executor.execute(() -> {
                log.warn("Execute outbox method {} without active transaction. It can lead to unexpected behaviour",
                    signature.getMethod());
                try {
                    repository.save(entity);
                    log.debug("Invoking outbox method: {}, args: {}", signature.getMethod(), pjp.getArgs());
                    methodInvoker.invoke();
                    log.debug("Outbox method {} successfully invoked", signature.getMethod());
                    repository.markCompleted(entity);
                    log.debug("Outbox method {} marked as completed", signature.getMethod());
                } catch (Exception e) {
                    log.error("Outbox method {} invoked with error", signature.getMethod(), e);
                }
            });
        }
        return null;
    }
}
