package com.github.kuliginstepan.outbox.core;

import java.lang.reflect.Method;
import java.time.Duration;
import java.time.Instant;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.ListableBeanFactory;
import org.springframework.util.ReflectionUtils;
import reactor.core.publisher.Mono;

@Slf4j
@RequiredArgsConstructor
public class DefaultReactiveOutboxRepublisher implements ReactiveOutboxRepublisher, BeanFactoryAware {

    private final ReactiveOutboxRepository repository;
    private final ReactiveRepublisherCallback callback;
    private final Duration notAfterPolicy;
    private ListableBeanFactory beanFactory;

    @Override
    public Mono<Void> republish() {
        return callback.beforeRepublish()
            .then(Mono.defer(() ->
                repository.findUncompletedEntities(Instant.now().minusMillis(notAfterPolicy.toMillis()))
                    .concatMap(entity -> republishEntity(entity, callback))
                    .then())
            )
            .then(Mono.defer(callback::afterRepublish));
    }

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = (ListableBeanFactory) beanFactory;
    }

    private Mono<Void> republishEntity(OutboxEntity entity, ReactiveRepublisherCallback callback) {
        return callback.beforeEntityRepublish(entity)
            .then(Mono.defer(() -> {
                OutboxMethodIdentifier identifier = entity.getMethodIdentifier();
                try {
                    Method method = ReflectionUtils.findMethod(identifier.getOutboxClass(),
                        identifier.getMethodName(),
                        identifier.getParameterTypes());
                    Object bean = beanFactory.getBean(identifier.getOutboxClass());
                    MethodInvoker methodInvoker = new MethodInvoker(method,
                        AopProxyUtils.getSingletonTarget(bean), entity.getData());
                    return ((Mono<?>) methodInvoker.invoke())
                        .then(Mono.defer(() -> repository.markCompleted(entity)))
                        .then(Mono.defer(() -> callback.afterEntityRepublish(entity)))
                        .onErrorResume(Exception.class, e -> {
                            log.error("Outbox method {} error", identifier.getValue(), e);
                            return callback.onEntityRepublishException(entity, e);
                        });
                } catch (Exception e) {
                    log.error("Outbox method {} error", identifier.getValue(), e);
                    return callback.onEntityRepublishException(entity, e);
                }
            }));
    }
}
