package com.github.kuliginstepan.outbox.core;

import java.lang.reflect.Method;
import java.time.Duration;
import java.time.Instant;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.ListableBeanFactory;
import org.springframework.util.ReflectionUtils;

@Slf4j
@RequiredArgsConstructor
public class DefaultOutboxRepublisher implements OutboxRepublisher, BeanFactoryAware {

    private final OutboxRepository repository;
    private final RepublisherCallback callback;
    private final Duration notAfterPolicy;
    private ListableBeanFactory beanFactory;

    public void republish() {
        callback.beforeRepublish();
        repository.findUncompletedEntities(Instant.now().minusMillis(notAfterPolicy.toMillis())).forEach(entity -> {
            callback.beforeEntityRepublish(entity);
            OutboxMethodIdentifier identifier = entity.getMethodIdentifier();
            try {
                Method method = ReflectionUtils.findMethod(identifier.getOutboxClass(), identifier.getMethodName(),
                    identifier.getParameterTypes());
                Object bean = beanFactory.getBean(identifier.getOutboxClass());
                new MethodInvoker(method, AopProxyUtils.getSingletonTarget(bean), entity.getData()).invoke();
                repository.markCompleted(entity);
                callback.afterEntityRepublish(entity);
            } catch (Exception e) {
                log.error("Outbox method {} error", identifier.getValue(), e);
                callback.onEntityRepublishException(entity, e);
            }
        });
        callback.afterRepublish();
    }

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = (ListableBeanFactory) beanFactory;
    }
}
