package com.github.kancyframework.springx.minidb;

import com.github.kancyframework.springx.log.Logger;
import com.github.kancyframework.springx.log.LoggerFactory;
import com.github.kancyframework.springx.minidb.serialize.ObjectDataSerializer;
import com.github.kancyframework.springx.utils.BeanUtils;
import com.github.kancyframework.springx.utils.FileUtils;
import com.github.kancyframework.springx.utils.PathUtils;
import com.github.kancyframework.springx.utils.SpringUtils;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * FileManager
 *
 * @author kancy
 * @date 2021/1/8 22:14
 */
public class ObjectDataManager {

    private static final Logger log = LoggerFactory.getLogger(ObjectDataManager.class);

    private static final String DEFAULT_NS = "springx-boot/minidb";

    private static final String DEFAULT_APP_NAME = System.getProperty("spring.application.name", "default");

    private static final Map<String, ObjectConfig> OBJECT_CACHE = new ConcurrentHashMap<>();

    private static final ObjectDataSerializer OBJECT_DATA_SERIALIZER = ObjectDataSerializerHolder.get();

    private ObjectDataManager() {
    }
    public static void flush() {
        flush(false);
    }
    public static void flush(boolean shutdown) {
        synchronized (OBJECT_CACHE){
            OBJECT_CACHE.values().forEach(objectConfig -> {
                try {
                    objectConfig.save();
                    if (shutdown){
                        log.info("minidb shutdown auto save : {}", objectConfig.getClass().getName());
                    } else {
                        log.debug("minidb auto save : {}", objectConfig.getClass().getName());
                    }
                } catch (Exception e) {
                    log.warn("minidb auto save fail : {} , config = {}", e.getMessage(),
                            objectConfig.getClass().getName());
                }
            });
            if (shutdown){
                // 关机后，清除数据
                OBJECT_CACHE.clear();
            }
        }
    }


    public static void store(ObjectConfig config) {
        String basePath = getUserHome();
        String objectId = config.getId();
        String path = PathUtils.path(basePath, DEFAULT_NS, objectId);

        File dataFile = null;
        try {
            dataFile = FileUtils.createNewFile(path);
            ObjectConfig object = OBJECT_CACHE.get(objectId);
            if (Objects.isNull(object)){
                OBJECT_CACHE.put(objectId, config);
                object = config;
            }
            synchronized (object){
                if (ObjectDataService.isProxy(object)){
                    Class<ObjectConfig> superclass = (Class<ObjectConfig>) object.getClass().getSuperclass();
                    ObjectConfig newObject = BeanUtils.copyProperties(object, superclass);
                    OBJECT_DATA_SERIALIZER.write(newObject, Files.newOutputStream(dataFile.toPath()));
                } else {
                    OBJECT_DATA_SERIALIZER.write(object, Files.newOutputStream(dataFile.toPath()));
                }
            }
        } catch (Exception e) {
            FileUtils.deleteFile(dataFile);
            throw new RuntimeException(e);
        }
    }

    public static <T extends ObjectConfig> T load(Class<T> dataClass) {
        return load(SpringUtils.getApplicationName(DEFAULT_APP_NAME), dataClass);
    }

    public static <T extends ObjectConfig> T loadByApplicationId(String applicationId, Class<T> dataClass) {
        return load(SpringUtils.getApplicationName(applicationId, DEFAULT_APP_NAME), dataClass);
    }

    private static <T extends ObjectConfig> T load(String appName, Class<T> dataClass) {
        String basePath = getUserHome();
        String dataFileName = getDataFileName(appName, dataClass);
        String objectId = appName + "/" + dataFileName;

        T data = null;
        if (OBJECT_CACHE.containsKey(objectId)){
            data = (T) OBJECT_CACHE.get(objectId);
            data.setId(objectId);
            log.info("命中[ObjectData]对象数据缓存：{}@{}", dataClass.getName(), data.hashCode());
            return data;
        }

        File dataFile = null;
        try {
            dataFile = new File(PathUtils.path(basePath, DEFAULT_NS, objectId));
            if (dataFile.exists() && dataFile.isFile()){
                data = OBJECT_DATA_SERIALIZER.read(Files.newInputStream(dataFile.toPath()), dataClass);
                // 如果使用代理模式
                if (ObjectDataService.isUseProxy(dataClass)){
                    T proxyData = ObjectDataService.initObjectData(dataClass);
                    BeanUtils.copyProperties(data, proxyData);
                    data = proxyData;
                }
            } else {
                data = ObjectDataService.initObjectData(dataClass);
                data.setId(objectId);
            }
        } catch (IOException e) {
            FileUtils.deleteFile(dataFile);
            throw new RuntimeException(e);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        // 放入缓存
        OBJECT_CACHE.put(objectId, data);
        return data;
    }

    private static String getDataFileName(String appName, Class<?> dataClass) {
        if (ObjectDataService.isProxy(dataClass)){
            dataClass = dataClass.getSuperclass();
        }
        String serializableType = OBJECT_DATA_SERIALIZER.getSerializableType();
        return md5(String.format("%s_%s_%s", appName, serializableType, dataClass.getName()));
    }

    private static String md5(String buffer) {
        String string = null;
        char hexDigist[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};
        MessageDigest md;
        try {
            md = MessageDigest.getInstance("MD5");
            md.update(buffer.getBytes());
            //16个字节的长整数
            byte[] datas = md.digest();

            char[] str = new char[2 * 16];
            int k = 0;

            for (int i = 0; i < 16; i++) {
                byte b = datas[i];
                //高4位
                str[k++] = hexDigist[b >>> 4 & 0xf];
                //低4位
                str[k++] = hexDigist[b & 0xf];
            }
            string = new String(str);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return string;
    }

    private static String getUserHome(){
        return System.getProperty("user.home");
    }

}
