/*
 * Decompiled with CFR 0.152.
 */
package com.github.iotexproject.mobile.solidity;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import com.github.iotexproject.mobile.account.IotexAccount;
import com.github.iotexproject.mobile.crypto.Bech32;
import com.github.iotexproject.mobile.utils.Numeric;
import java.lang.reflect.Array;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public abstract class SolidityType {
    private static final int Int32Size = 32;
    protected String name;

    public SolidityType(String name) {
        this.name = name;
    }

    @JsonCreator
    public static SolidityType getType(String typeName) {
        if (typeName.endsWith("]")) {
            return ArrayType.getType(typeName);
        }
        if ("bool".equals(typeName)) {
            return new BoolType();
        }
        if (typeName.startsWith("int")) {
            return new IntType(typeName);
        }
        if (typeName.startsWith("uint")) {
            return new UnsignedIntType(typeName);
        }
        if ("address".equals(typeName)) {
            return new AddressType();
        }
        if ("string".equals(typeName)) {
            return new StringType();
        }
        if ("bytes".equals(typeName)) {
            return new BytesType();
        }
        if ("function".equals(typeName)) {
            return new FunctionType();
        }
        if (typeName.startsWith("bytes")) {
            return new Bytes32Type(typeName);
        }
        throw new RuntimeException("Unknown type: " + typeName);
    }

    public String getName() {
        return this.name;
    }

    @JsonValue
    public String getCanonicalName() {
        return this.getName();
    }

    public abstract byte[] encode(Object var1);

    public abstract Object decode(byte[] var1, int var2);

    public Object decode(byte[] encoded) {
        return this.decode(encoded, 0);
    }

    public int getFixedSize() {
        return 32;
    }

    public boolean isDynamicType() {
        return false;
    }

    public String toString() {
        return this.getName();
    }

    public static class FunctionType
    extends Bytes32Type {
        public FunctionType() {
            super("function");
        }

        @Override
        public byte[] encode(Object value) {
            if (!(value instanceof byte[])) {
                throw new RuntimeException("Expected byte[] value for FunctionType");
            }
            if (((byte[])value).length != 24) {
                throw new RuntimeException("Expected byte[24] for FunctionType");
            }
            return super.encode(Numeric.merge((byte[])value, new byte[8]));
        }
    }

    public static class BoolType
    extends IntType {
        public BoolType() {
            super("bool");
        }

        @Override
        public byte[] encode(Object value) {
            if (!(value instanceof Boolean)) {
                throw new RuntimeException("Wrong value for bool type: " + value);
            }
            return super.encode(value == Boolean.TRUE ? 1 : 0);
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            return ((Number)super.decode(encoded, offset)).intValue() != 0;
        }
    }

    public static class UnsignedIntType
    extends NumericType {
        public UnsignedIntType(String name) {
            super(name);
        }

        public static BigInteger decodeInt(byte[] encoded, int offset) {
            return new BigInteger(1, Arrays.copyOfRange(encoded, offset, offset + 32));
        }

        public static byte[] encodeInt(int i) {
            return UnsignedIntType.encodeInt(new BigInteger("" + i));
        }

        public static byte[] encodeInt(BigInteger bigInt) {
            if (bigInt.signum() == -1) {
                throw new RuntimeException("Wrong value for uint type: " + bigInt);
            }
            return Numeric.bigIntegerToBytes(bigInt, 32);
        }

        @Override
        public String getCanonicalName() {
            if (this.getName().equals("uint")) {
                return "uint256";
            }
            return super.getCanonicalName();
        }

        @Override
        public byte[] encode(Object value) {
            BigInteger bigInt = this.encodeInternal(value);
            return UnsignedIntType.encodeInt(bigInt);
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            return UnsignedIntType.decodeInt(encoded, offset);
        }
    }

    public static class IntType
    extends NumericType {
        public IntType(String name) {
            super(name);
        }

        public static BigInteger decodeInt(byte[] encoded, int offset) {
            return new BigInteger(Arrays.copyOfRange(encoded, offset, offset + 32));
        }

        public static byte[] encodeInt(int i) {
            return IntType.encodeInt(new BigInteger("" + i));
        }

        public static byte[] encodeInt(BigInteger bigInt) {
            return Numeric.bigIntegerToBytesSigned(bigInt, 32);
        }

        @Override
        public String getCanonicalName() {
            if (this.getName().equals("int")) {
                return "int256";
            }
            return super.getCanonicalName();
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            return IntType.decodeInt(encoded, offset);
        }

        @Override
        public byte[] encode(Object value) {
            BigInteger bigInt = this.encodeInternal(value);
            return IntType.encodeInt(bigInt);
        }
    }

    public static abstract class NumericType
    extends SolidityType {
        public NumericType(String name) {
            super(name);
        }

        BigInteger encodeInternal(Object value) {
            BigInteger bigInt;
            if (value instanceof String) {
                String s = ((String)value).toLowerCase().trim();
                int radix = 10;
                if (s.startsWith("0x")) {
                    s = s.substring(2);
                    radix = 16;
                } else if (s.contains("a") || s.contains("b") || s.contains("c") || s.contains("d") || s.contains("e") || s.contains("f")) {
                    radix = 16;
                }
                bigInt = new BigInteger(s, radix);
            } else if (value instanceof BigInteger) {
                bigInt = (BigInteger)value;
            } else if (value instanceof Number) {
                bigInt = new BigInteger(value.toString());
            } else if (value instanceof byte[]) {
                bigInt = Numeric.toBigInt((byte[])value);
            } else {
                throw new RuntimeException("Invalid value for type '" + this + "': " + value + " (" + value.getClass() + ")");
            }
            return bigInt;
        }
    }

    public static class AddressType
    extends IntType {
        public AddressType() {
            super("address");
        }

        @Override
        public byte[] encode(Object value) {
            if (value instanceof String && !((String)value).startsWith("io")) {
                value = "io" + value;
            }
            value = IotexAccount.convertToETHAddress(value.toString());
            byte[] addr = super.encode(value);
            for (int i = 0; i < 12; ++i) {
                if (addr[i] == 0) continue;
                throw new RuntimeException("Invalid address (should be 20 bytes length): " + Numeric.toHexString(addr));
            }
            return addr;
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            BigInteger bi = (BigInteger)super.decode(encoded, offset);
            byte[] values = Numeric.bigIntegerToBytes(bi, 20);
            byte[] grouped = Bech32.convertBits(values, 0, values.length, 8, 5, true);
            return Bech32.encode("io", grouped);
        }
    }

    public static class Bytes32Type
    extends SolidityType {
        public Bytes32Type(String s) {
            super(s);
        }

        public static byte[] decodeBytes32(byte[] encoded, int offset) {
            return Arrays.copyOfRange(encoded, offset, offset + 32);
        }

        @Override
        public byte[] encode(Object value) {
            if (value instanceof Number) {
                BigInteger bigInt = new BigInteger(value.toString());
                return IntType.encodeInt(bigInt);
            }
            if (value instanceof String) {
                byte[] ret = new byte[32];
                byte[] bytes = ((String)value).getBytes(StandardCharsets.UTF_8);
                System.arraycopy(bytes, 0, ret, 0, bytes.length);
                return ret;
            }
            if (value instanceof byte[]) {
                byte[] bytes = (byte[])value;
                byte[] ret = new byte[32];
                System.arraycopy(bytes, 0, ret, 32 - bytes.length, bytes.length);
                return ret;
            }
            throw new RuntimeException("Can't encode java type " + value.getClass() + " to bytes32");
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            return Bytes32Type.decodeBytes32(encoded, offset);
        }
    }

    public static class StringType
    extends BytesType {
        public StringType() {
            super("string");
        }

        @Override
        public byte[] encode(Object value) {
            if (!(value instanceof String)) {
                throw new RuntimeException("String value expected for type 'string'");
            }
            return super.encode(((String)value).getBytes(StandardCharsets.UTF_8));
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            return new String((byte[])super.decode(encoded, offset), StandardCharsets.UTF_8);
        }
    }

    public static class BytesType
    extends SolidityType {
        protected BytesType(String name) {
            super(name);
        }

        public BytesType() {
            super("bytes");
        }

        @Override
        public byte[] encode(Object value) {
            byte[] bb;
            if (value instanceof byte[]) {
                bb = (byte[])value;
            } else if (value instanceof String) {
                bb = ((String)value).getBytes();
            } else {
                throw new RuntimeException("byte[] or String value is expected for type 'bytes'");
            }
            byte[] ret = new byte[((bb.length - 1) / 32 + 1) * 32];
            System.arraycopy(bb, 0, ret, 0, bb.length);
            return Numeric.merge(IntType.encodeInt(bb.length), ret);
        }

        @Override
        public Object decode(byte[] encoded, int offset) {
            int len = IntType.decodeInt(encoded, offset).intValue();
            if (len == 0) {
                return new byte[0];
            }
            return Arrays.copyOfRange(encoded, offset += 32, offset + len);
        }

        @Override
        public boolean isDynamicType() {
            return true;
        }
    }

    public static class DynamicArrayType
    extends ArrayType {
        public DynamicArrayType(String name) {
            super(name);
        }

        @Override
        public String getCanonicalName() {
            return this.elementType.getCanonicalName() + "[]";
        }

        @Override
        public byte[] encodeList(List l) {
            return Numeric.merge(IntType.encodeInt(l.size()), this.encodeTuple(l));
        }

        @Override
        public Object decode(byte[] encoded, int origOffset) {
            int len = IntType.decodeInt(encoded, origOffset).intValue();
            return this.decodeTuple(encoded, origOffset + 32, len);
        }

        @Override
        public boolean isDynamicType() {
            return true;
        }
    }

    public static class StaticArrayType
    extends ArrayType {
        int size;

        public StaticArrayType(String name) {
            super(name);
            int idx1 = name.lastIndexOf("[");
            int idx2 = name.lastIndexOf("]");
            String dim = name.substring(idx1 + 1, idx2);
            this.size = Integer.parseInt(dim);
        }

        @Override
        public String getCanonicalName() {
            return this.getElementType().getCanonicalName() + "[" + this.size + "]";
        }

        @Override
        public byte[] encodeList(List l) {
            if (l.size() != this.size) {
                throw new RuntimeException("List size (" + l.size() + ") != " + this.size + " for type " + this.getName());
            }
            return this.encodeTuple(l);
        }

        public Object[] decode(byte[] encoded, int offset) {
            return this.decodeTuple(encoded, offset, this.size);
        }

        @Override
        public int getFixedSize() {
            if (this.isDynamicType()) {
                return 32;
            }
            return this.elementType.getFixedSize() * this.size;
        }

        @Override
        public boolean isDynamicType() {
            return this.getElementType().isDynamicType() && this.size > 0;
        }
    }

    public static abstract class ArrayType
    extends SolidityType {
        SolidityType elementType;

        public ArrayType(String name) {
            super(name);
            this.elementType = SolidityType.getType(name.substring(0, name.lastIndexOf("[")));
        }

        public static ArrayType getType(String typeName) {
            int idx2;
            int idx1 = typeName.lastIndexOf("[");
            if (idx1 + 1 == (idx2 = typeName.lastIndexOf("]"))) {
                return new DynamicArrayType(typeName);
            }
            return new StaticArrayType(typeName);
        }

        @Override
        public byte[] encode(Object value) {
            if (value.getClass().isArray()) {
                ArrayList<Object> elems = new ArrayList<Object>();
                for (int i = 0; i < Array.getLength(value); ++i) {
                    elems.add(Array.get(value, i));
                }
                return this.encodeList(elems);
            }
            if (value instanceof List) {
                return this.encodeList((List)value);
            }
            throw new RuntimeException("List value expected for type " + this.getName());
        }

        protected byte[] encodeTuple(List l) {
            byte[][] elems;
            if (this.elementType.isDynamicType()) {
                elems = new byte[l.size() * 2][];
                int offset = l.size() * 32;
                for (int i = 0; i < l.size(); ++i) {
                    elems[i] = IntType.encodeInt(offset);
                    byte[] encoded = this.elementType.encode(l.get(i));
                    elems[l.size() + i] = encoded;
                    offset += 32 * ((encoded.length - 1) / 32 + 1);
                }
            } else {
                elems = new byte[l.size()][];
                for (int i = 0; i < l.size(); ++i) {
                    elems[i] = this.elementType.encode(l.get(i));
                }
            }
            return Numeric.merge(elems);
        }

        public Object[] decodeTuple(byte[] encoded, int origOffset, int len) {
            int offset = origOffset;
            Object[] ret = new Object[len];
            for (int i = 0; i < len; ++i) {
                ret[i] = this.elementType.isDynamicType() ? this.elementType.decode(encoded, origOffset + IntType.decodeInt(encoded, offset).intValue()) : this.elementType.decode(encoded, offset);
                offset += this.elementType.getFixedSize();
            }
            return ret;
        }

        public SolidityType getElementType() {
            return this.elementType;
        }

        public abstract byte[] encodeList(List var1);
    }
}

