package fun.gen;

import java.util.function.Supplier;
import java.util.random.RandomGenerator;

import static java.util.Objects.requireNonNull;

/**
 * The {@code NamedGen} class represents a named generator that can be used to create and manage
 * generators in a way that supports laziness and recursion.
 *
 * <p>Note: When using the {@link #of(String, Gen)} method to associate a generator with a name, be aware that
 * attempting to generate data using the named generator without a prior association may result in a
 * {@code RuntimeException}. Ensure that a generator has been properly associated with the name using the mentioned method.
 *
 * @param <O> The type of values generated by the named generator.
 */
public final class NamedGen<O> implements Gen<O> {

    final String name;

    private NamedGen(final String name) {
        this.name = requireNonNull(name);
    }

    /**
     * Creates a new instance of a named generator with the specified name.
     *
     * @param <O>  The type of values generated by the named generator.
     * @param name The name associated with the generator.
     * @return A new named generator with the specified name.
     * @throws NullPointerException If the provided name is {@code null}.
     */
    public static <O> Gen<O> of(final String name) {
        return new NamedGen<>(name);
    }

    /**
     * Creates a new instance of a named generator with the specified name and associates it with
     * the provided generator. This method also stores the association in the generator cache.
     *
     * @param <O>  The type of values generated by the named generator.
     * @param name The name associated with the generator.
     * @param gen  The generator to associate with the specified name.
     * @return A new named generator with the specified name.
     * @throws NullPointerException     If the provided name or generator is {@code null}.
     * @throws IllegalArgumentException If a generator with the same name already exists in the cache.
     */
    public static <O> Gen<O> of(final String name,
                                final Gen<O> gen) {
        GenCache.put(name,
                     gen);
        return new NamedGen<>(name);
    }

    /**
     * Applies the named generator to generate values based on a random source.
     * This method uses laziness and recursion to support generators that reference each other.
     *
     * @param random The random source used for generating values.
     * @return A supplier that produces values using the named generator.
     * @throws NullPointerException     If the provided random source is {@code null}.
     * @throws IllegalArgumentException If the named generator with the specified name does not exist in the cache.
     * @throws RuntimeException         If no generator has been associated with the specified name using the {@link  #of(String, Gen)}  method.
     */
    @Override
    public Supplier<O> apply(RandomGenerator random) {
        requireNonNull(random);
        //we need another level of lazyness here to support recursive generators
        return () -> GenCache.<Gen<O>>get(name)
                             .apply(random)
                             .get();
    }
}
