package com.github.houbb.jdbc.api.dal;

import com.github.houbb.heaven.support.handler.IHandler;
import com.github.houbb.jdbc.api.support.ITypeHandlerRegister;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

/**
 * JDBC 访问层
 * @author bbhou
 * date 2017/8/1
 * @since 0.0.1
 */
public interface IMapper {

    /**
     * 查询单个信息
     * 反射原理：为了更便于阅读，基于 {@link ResultSet#getObject(String)} 根据列名称获取。
     *
     * 默认返回第一个
     * @param querySql 查询的脚本
     * @param tClass 对象信息
     * @param <T> 泛型
     * @return 结果
     * @see com.github.houbb.jdbc.api.annotation.Column 列信息
     */
    <T> T selectOne(String querySql, final Class<T> tClass);

    /**
     * 查询列表信息
     * @param querySql 查询的脚本
     * @param tClass 对象信息
     * @param <T> 泛型
     * @return 结果列表
     * @see com.github.houbb.jdbc.api.annotation.Column 列信息
     */
    <T> List<T> selectList(String querySql, final Class<T> tClass);

    /**
     * 查询列表信息
     * @param querySql 查询的脚本
     * @param tClass 对象信息
     * @param register 注册类型
     * @param <T> 泛型
     * @return 结果列表
     * @see com.github.houbb.jdbc.api.annotation.Column 列信息
     * @since 0.0.3
     */
    <T> List<T> selectList(String querySql, Class<T> tClass, ITypeHandlerRegister register);

    /**
     * 对查询结果执行转换-查询单个
     *
     * 默认返回第一个
     * @param sql 查询的脚本
     * @param handler 处理转换器
     * @param <T> 泛型
     * @return 转换后的结果
     */
    <T> T selectOne(final String sql, final IHandler<ResultSet, T> handler);

    /**
     * 对查询结果执行转换-查询列表
     * @param sql 查询的脚本
     * @param handler 处理转换器
     * @param <T> 泛型
     * @return 转换后的结果列表
     */
    <T> List<T> selectList(final String sql, final IHandler<ResultSet, T> handler);

    /**
     * 对查询结果执行转换-查询列表
     * @param prepareSql 查询的脚本
     * @param handler 处理转换器
     * @param prepareInfos 数据列表
     * @param <T> 泛型
     * @return 转换后的结果列表
     * @since 0.0.3
     */
    <T> List<T> selectList(final String prepareSql, final IHandler<ResultSet, T> handler, final List<IPrepareInfo> prepareInfos);

    /**
     * 执行SQL
     * @param sql 脚本
     * @return 执行结果
     */
    boolean execute(String sql);

    /**
     * 执行SQL
     * @param prepareSql 准备脚本
     * @param prepareInfos 对应信息
     * @return 执行结果
     * @since 0.0.3
     */
    boolean execute(String prepareSql, List<IPrepareInfo> prepareInfos);

    /**
     * 批量执行脚本；不包含事物。
     * @param stringList 语句列表
     * @return 共计影响的条数
     */
    int executeBatch(List<String> stringList);

    /**
     * 执行SQL
     * 1.包含事物处理
     * 2. sql 本身包含多条sql。同时成功或者失败。
     * @param stringList 入参列表
     * @throws SQLException 异常
     * @since 0.0.1
     * @return 结果
     */
    int executeBatchTx(List<String> stringList) throws SQLException;

    /**
     * 数据库元始数据信息
     * @return 数据库元数据信息
     */
    DatabaseMetaData metadata();

    /**
     * 获取数据库名称
     * @return 数据库名称
     */
    String databaseName();

}
