package com.github.faubertin.http.client.facade;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpVersion;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import com.github.faubertin.http.client.domain.HttpRequest;
import com.github.faubertin.http.client.domain.HttpResponse;
import com.github.faubertin.http.client.handler.HttpClientHandler;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.function.Consumer;

import static io.netty.handler.codec.http.HttpHeaders.Names.*;
import static io.netty.handler.codec.http.HttpHeaders.Values.CLOSE;
import static io.netty.handler.codec.http.HttpHeaders.Values.GZIP;

@Slf4j
@RequiredArgsConstructor
public class SimpleHttpClient implements HttpClient {

    private final Bootstrap nettyBootstrap;

    @Override
    public void get(HttpRequest httpRequest,
                    Consumer<HttpResponse> callback) {
        sendHttpRequest(
                HttpMethod.GET,
                httpRequest,
                callback);
    }

    @Override
    public void post(HttpRequest httpRequest,
                     Consumer<HttpResponse> callback) {
        sendHttpRequest(
                HttpMethod.POST,
                httpRequest,
                callback);
    }

    @Override
    public void put(HttpRequest httpRequest,
                    Consumer<HttpResponse> callback) {
        sendHttpRequest(
                HttpMethod.PUT,
                httpRequest,
                callback);
    }

    void sendHttpRequest(HttpMethod method,
                         HttpRequest httpRequest,
                         Consumer<HttpResponse> callback) {
        try {
            sendHttpRequestInterruptibly(method, httpRequest, callback);
        } catch (InterruptedException e) {
            log.warn("Thread interrupted, aborting HTTP request");
            Thread.currentThread().interrupt();
        }
    }

    void sendHttpRequestInterruptibly(HttpMethod method,
                                      HttpRequest httpRequest,
                                      Consumer<HttpResponse> callback) throws InterruptedException {
        URI uri = parseUri(httpRequest.getUrl());
        Channel connection = createConnection(uri.getHost(), getPort(uri));

        addCallbackInvocationToPipeline(connection, callback);

        FullHttpRequest nettyHttpRequest = createNettyHttpRequest(method, uri);
        writeAndFlush(connection, nettyHttpRequest);
    }

    Channel createConnection(String host,
                             int port) throws InterruptedException {
        return nettyBootstrap.connect(host, port)
                .sync()
                .channel();
    }

    void addCallbackInvocationToPipeline(Channel connection,
                                         Consumer<HttpResponse> callback) {
        connection.pipeline()
                .addLast("http-client", new HttpClientHandler((HttpResponse httpResponse) -> {
                    closeConnectionSafely(connection);
                    callback.accept(httpResponse);
                }));
    }

    FullHttpRequest createNettyHttpRequest(HttpMethod method, URI uri) {
        DefaultFullHttpRequest httpRequest = new DefaultFullHttpRequest(
                HttpVersion.HTTP_1_1,
                method,
                uri.getRawPath());
        httpRequest.headers().set(HOST, uri.getHost());
        httpRequest.headers().set(CONNECTION, CLOSE);
        httpRequest.headers().set(ACCEPT_ENCODING, GZIP);
        return httpRequest;
    }

    void writeAndFlush(Channel connection, FullHttpRequest httpRequest) {
        connection.writeAndFlush(httpRequest)
                .addListener((ChannelFuture channelFuture) -> {
                    if (!channelFuture.isSuccess()) {
                        log.error("Error while sending HTTP request", channelFuture.cause());
                    }
                });
    }

    void closeConnectionSafely(Channel connection) {
        try {
            connection.close()
                    .addListener((ChannelFuture channelFuture) -> {
                        if (!channelFuture.isSuccess()) {
                            log.error("Error while closing connection", channelFuture.cause());
                        }
                    });
        } catch (Exception e) {
            log.error("Error while closing HTTP connection", e);
        }
    }

    URI parseUri(String url) {
        try {
            return new URI(url);
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Invalid URL", e);
        }
    }

    int getPort(URI uri) {
        int port = uri.getPort();
        if (port > 0) {
            return port;
        }
        switch(uri.getScheme()) {
            case "http":
                return 80;
            case "https":
                return 443;
            default:
                throw new IllegalArgumentException("Unknown protocol: " + uri.getScheme());
        }
    }

}
