package com.github.faubertin.http.client.domain;

import com.google.common.collect.ImmutableMap;
import lombok.Data;
import lombok.NonNull;

import java.util.Map;
import java.util.Optional;

import static com.google.common.base.Preconditions.checkArgument;

@Data
public class HttpRequest {

    public static HttpRequestBuilder newBuilder(String url) {
        return new HttpRequestBuilder(url);
    }

    private final String url;
    private final Optional<byte[]> body;
    private final ImmutableMap<String, String> headers;

    private HttpRequest(
            String url,
            Optional<byte[]> body,
            ImmutableMap<String, String> headers) {
        this.url = url;
        this.body = body;
        this.headers = headers;
    }

    public static class HttpRequestBuilder {

        @NonNull
        private final String url;
        private byte[] body;
        private ImmutableMap.Builder<String, String> headers;

        private HttpRequestBuilder(String url) {
            checkArgument(!url.trim().isEmpty(), "Blank URL not allowed");
            this.url = url;
            this.headers = ImmutableMap.builder();
        }

        public HttpRequestBuilder withBody(@NonNull byte[] body) {
            this.body = body;
            return this;
        }

        public HttpRequestBuilder withHeader(@NonNull String name,
                                             @NonNull String value) {
            checkArgument(!name.trim().isEmpty(), "Blank header name not allowed");
            this.headers.put(name, value);
            return this;
        }

        public HttpRequestBuilder withHeaders(@NonNull Map<String, String> headers) {
            this.headers.putAll(headers);
            return this;
        }

        public HttpRequest build() {
            return new HttpRequest(
                    url,
                    Optional.ofNullable(body),
                    headers.build());
        }

    }

}
