/*
 * Decompiled with CFR 0.152.
 */
package com.github.davidmoten.rx2.file;

import com.github.davidmoten.guavamini.Lists;
import com.github.davidmoten.guavamini.Preconditions;
import com.github.davidmoten.rx2.Bytes;
import com.github.davidmoten.rx2.Strings;
import io.reactivex.Flowable;
import io.reactivex.Scheduler;
import io.reactivex.flowables.GroupedFlowable;
import io.reactivex.functions.Function;
import io.reactivex.functions.Predicate;
import io.reactivex.schedulers.Schedulers;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

public final class Files {
    private static final int DEFAULT_POLLING_INTERVAL_MS = 1000;
    public static final int DEFAULT_MAX_BYTES_PER_EMISSION = 8192;
    public static final List<WatchEvent.Kind<?>> ALL_KINDS = Lists.newArrayList((Object[])new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.OVERFLOW});
    private static Function<Object, Boolean> IS_MODIFY_OR_OVERFLOW = event -> {
        if (event instanceof WatchEvent) {
            WatchEvent w = (WatchEvent)event;
            String kind = w.kind().name();
            if (kind.equals(StandardWatchEventKinds.ENTRY_MODIFY.name()) || kind.equals(StandardWatchEventKinds.OVERFLOW.name())) {
                return true;
            }
            return false;
        }
        return false;
    };

    private Files() {
    }

    private static Flowable<byte[]> tailBytes(File file, long startPosition, long pollingIntervalMs, int chunkSize, Flowable<?> events) {
        Preconditions.checkNotNull((Object)file);
        return Files.sampleModifyOrOverflowEventsOnly(events, pollingIntervalMs * 2L).compose(x -> Files.eventsToBytes(events, file, startPosition, chunkSize));
    }

    private static Flowable<String> tailLines(File file, long startPosition, int chunkSize, Charset charset, Flowable<?> events) {
        Preconditions.checkNotNull((Object)file);
        Preconditions.checkNotNull((Object)charset);
        Preconditions.checkNotNull(events);
        return Files.toLines((Flowable<byte[]>)events.compose(x -> Files.eventsToBytes(x, file, startPosition, chunkSize)), charset);
    }

    private static Flowable<WatchEvent<?>> events(WatchService watchService, Scheduler scheduler, long intervalMs) {
        Preconditions.checkNotNull((Object)watchService, (String)"watchService cannot be null");
        Preconditions.checkNotNull((Object)scheduler, (String)"scheduler cannot be null");
        Preconditions.checkArgument((intervalMs > 0L ? 1 : 0) != 0, (String)"intervalMs must be positive");
        return Flowable.interval((long)intervalMs, (TimeUnit)TimeUnit.MILLISECONDS, (Scheduler)scheduler).flatMap(x -> {
            try {
                WatchKey key = watchService.poll();
                if (key != null) {
                    Flowable r = Flowable.fromIterable(key.pollEvents());
                    key.reset();
                    return r;
                }
                return Flowable.empty();
            }
            catch (ClosedWatchServiceException e) {
                return Flowable.empty();
            }
        });
    }

    private static Flowable<WatchEvent<?>> events(File file, Scheduler scheduler, long pollingIntervalMs, List<WatchEvent.Kind<?>> kinds, List<WatchEvent.Modifier> modifiers) {
        return Flowable.using(() -> Files.watchService(file, kinds, modifiers), ws -> Files.events(ws, scheduler, pollingIntervalMs).filter(Files.onlyRelatedTo(file)), ws -> ws.close(), (boolean)true);
    }

    private static WatchService watchService(File file, List<WatchEvent.Kind<?>> kinds, List<WatchEvent.Modifier> modifiers) throws IOException {
        Path path = Files.getBasePath(file);
        WatchService watchService = path.getFileSystem().newWatchService();
        path.register(watchService, kinds.toArray(new WatchEvent.Kind[0]), modifiers.toArray(new WatchEvent.Modifier[0]));
        return watchService;
    }

    private static final Predicate<WatchEvent<?>> onlyRelatedTo(final File file) {
        return new Predicate<WatchEvent<?>>(){

            public boolean test(WatchEvent<?> event) {
                boolean ok;
                if (file.isDirectory()) {
                    ok = true;
                } else if (StandardWatchEventKinds.OVERFLOW.equals(event.kind())) {
                    ok = true;
                } else {
                    Object context = event.context();
                    if (context != null && context instanceof Path) {
                        Path p = (Path)context;
                        Path basePath = Files.getBasePath(file);
                        File pFile = new File(basePath.toFile(), p.toString());
                        ok = pFile.getAbsolutePath().equals(file.getAbsolutePath());
                    } else {
                        ok = false;
                    }
                }
                return ok;
            }
        };
    }

    private static Flowable<String> toLines(Flowable<byte[]> bytes, Charset charset) {
        return Strings.split((Flowable)Strings.decode(bytes, (Charset)charset), (String)"\n");
    }

    private static Flowable<Object> sampleModifyOrOverflowEventsOnly(Flowable<?> events, long sampleTimeMs) {
        return events.groupBy(IS_MODIFY_OR_OVERFLOW).flatMap(Files.sampleIfTrue(sampleTimeMs));
    }

    private static Function<GroupedFlowable<Boolean, ?>, Flowable<?>> sampleIfTrue(long sampleTimeMs) {
        return group -> {
            if (((Boolean)group.getKey()).booleanValue()) {
                return group.sample(sampleTimeMs, TimeUnit.MILLISECONDS);
            }
            return group;
        };
    }

    public static WatchEventsBuilder events(File file) {
        return new WatchEventsBuilder(file);
    }

    public static TailBytesBuilder tailBytes(File file) {
        return new TailBytesBuilder(file);
    }

    public static TailBytesBuilder tailBytes(String filename) {
        return Files.tailBytes(new File(filename));
    }

    public static TailLinesBuilder tailLines(File file) {
        return new TailLinesBuilder(file);
    }

    public static TailLinesBuilder tailLines(String filename) {
        return Files.tailLines(new File(filename));
    }

    private static Flowable<byte[]> eventsToBytes(Flowable<?> events, File file, long startPosition, int chunkSize) {
        return Flowable.defer(() -> {
            State state = new State();
            state.position = startPosition;
            return events.flatMap(event -> Files.eventToBytes(event, file, state, chunkSize));
        });
    }

    private static Flowable<byte[]> eventToBytes(Object event, File file, State state, int chunkSize) {
        long length;
        if (event instanceof WatchEvent) {
            WatchEvent w = (WatchEvent)event;
            String kind = w.kind().name();
            if (kind.equals(StandardWatchEventKinds.ENTRY_CREATE.name())) {
                state.position = 0L;
            } else if (kind.equals(StandardWatchEventKinds.ENTRY_DELETE.name())) {
                return Flowable.error((Throwable)new IOException("file has been deleted"));
            }
        }
        if ((length = file.length()) > state.position) {
            return Flowable.using(() -> new FileInputStream(file), fis -> {
                fis.skip(state.position);
                return Bytes.from((InputStream)fis, (int)chunkSize).doOnNext(x -> state.position += (long)((byte[])x).length);
            }, fis -> fis.close(), (boolean)true);
        }
        return Flowable.empty();
    }

    private static final Path getBasePath(File file) {
        Path path = file.exists() && file.isDirectory() ? Paths.get(file.toURI()) : Paths.get(file.getParentFile().toURI());
        return path;
    }

    private static final class State {
        long position;

        private State() {
        }
    }

    public static final class TailLinesBuilder {
        private final File file;
        private long startPosition = 0L;
        private int chunkSize = 8192;
        private Charset charset = StandardCharsets.UTF_8;
        private long pollingIntervalMs = 1000L;
        private Scheduler scheduler = Schedulers.io();
        private Flowable<?> events;
        private final List<WatchEvent.Modifier> modifiers = new ArrayList<WatchEvent.Modifier>();

        TailLinesBuilder(File file) {
            this.file = file;
        }

        public TailLinesBuilder startPosition(long startPosition) {
            this.startPosition = startPosition;
            return this;
        }

        public TailLinesBuilder pollingInterval(long pollingInterval, TimeUnit unit) {
            this.pollingIntervalMs = unit.toMillis(pollingInterval);
            return this;
        }

        public TailLinesBuilder chunkSize(int chunkSize) {
            this.chunkSize = chunkSize;
            return this;
        }

        public TailLinesBuilder charset(Charset charset) {
            this.charset = charset;
            return this;
        }

        public TailLinesBuilder charset(String charset) {
            return this.charset(Charset.forName(charset));
        }

        public TailLinesBuilder utf8() {
            return this.charset("UTF-8");
        }

        public TailLinesBuilder scheduler(Scheduler scheduler) {
            this.scheduler = scheduler;
            return this;
        }

        public TailLinesBuilder modifier(WatchEvent.Modifier modifier) {
            this.modifiers.add(modifier);
            return this;
        }

        public TailLinesBuilder events(Flowable<?> events) {
            this.events = events;
            return this;
        }

        private Flowable<?> events() {
            if (this.events == null) {
                return Files.events(this.file, this.scheduler, this.pollingIntervalMs, ALL_KINDS, this.modifiers);
            }
            return this.events;
        }

        public Flowable<String> build() {
            return Files.tailLines(this.file, this.startPosition, this.chunkSize, this.charset, this.events());
        }
    }

    public static final class TailBytesBuilder {
        private final File file;
        private long startPosition = 0L;
        private int chunkSize = 8192;
        private long pollingIntervalMs = 1000L;
        private Scheduler scheduler = Schedulers.io();
        private Flowable<?> events;
        private final List<WatchEvent.Modifier> modifiers = new ArrayList<WatchEvent.Modifier>();

        TailBytesBuilder(File file) {
            this.file = file;
        }

        public TailBytesBuilder startPosition(long startPosition) {
            this.startPosition = startPosition;
            return this;
        }

        public TailBytesBuilder pollingInterval(long pollingInterval, TimeUnit unit) {
            this.pollingIntervalMs = unit.toMillis(pollingInterval);
            return this;
        }

        public TailBytesBuilder chunkSize(int chunkSize) {
            this.chunkSize = chunkSize;
            return this;
        }

        public TailBytesBuilder scheduler(Scheduler scheduler) {
            this.scheduler = scheduler;
            return this;
        }

        public TailBytesBuilder events(Flowable<?> events) {
            this.events = events;
            return this;
        }

        public TailBytesBuilder modifier(WatchEvent.Modifier modifier) {
            this.modifiers.add(modifier);
            return this;
        }

        private Flowable<?> events() {
            if (this.events == null) {
                return Files.events(this.file, this.scheduler, this.pollingIntervalMs, ALL_KINDS, this.modifiers);
            }
            return this.events;
        }

        public Flowable<byte[]> build() {
            return Files.tailBytes(this.file, this.startPosition, this.pollingIntervalMs, this.chunkSize, this.events());
        }
    }

    public static final class WatchEventsBuilder {
        private final File file;
        private Optional<Scheduler> scheduler = Optional.empty();
        private long pollInterval = 1000L;
        private TimeUnit pollIntervalUnit = TimeUnit.MILLISECONDS;
        private final List<WatchEvent.Kind<?>> kinds = new ArrayList();
        private final List<WatchEvent.Modifier> modifiers = new ArrayList<WatchEvent.Modifier>();

        private WatchEventsBuilder(File file) {
            this.file = file;
        }

        public WatchEventsBuilder scheduler(Scheduler scheduler) {
            this.scheduler = Optional.of(scheduler);
            return this;
        }

        public WatchEventsBuilder pollInterval(long interval, TimeUnit unit) {
            this.pollInterval = interval;
            this.pollIntervalUnit = unit;
            return this;
        }

        public WatchEventsBuilder kind(WatchEvent.Kind<?> kind) {
            this.kinds.add(kind);
            return this;
        }

        public WatchEventsBuilder modifier(WatchEvent.Modifier modifier) {
            this.modifiers.add(modifier);
            return this;
        }

        public WatchEventsBuilder kinds(WatchEvent.Kind<?> ... kinds) {
            for (WatchEvent.Kind<?> kind : kinds) {
                this.kinds.add(kind);
            }
            return this;
        }

        public Flowable<WatchEvent<?>> build() {
            ArrayList kindsCopy = new ArrayList(this.kinds);
            if (kindsCopy.isEmpty()) {
                kindsCopy.add(StandardWatchEventKinds.ENTRY_CREATE);
                kindsCopy.add(StandardWatchEventKinds.ENTRY_DELETE);
                kindsCopy.add(StandardWatchEventKinds.ENTRY_MODIFY);
                kindsCopy.add(StandardWatchEventKinds.OVERFLOW);
            }
            return Flowable.using(() -> Files.watchService(this.file, kindsCopy, this.modifiers), ws -> Files.events(ws, this.scheduler.orElse(Schedulers.io()), this.pollIntervalUnit.toMillis(this.pollInterval)), ws -> ws.close(), (boolean)true);
        }
    }
}

