package com.github.damianwajser.filter;

import com.github.damianwajser.configuration.MDCProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.Enumeration;
import java.util.UUID;

@Component
@Order(Ordered.HIGHEST_PRECEDENCE)
public abstract class MDCFilter implements Filter, MDCProperties {

	private static final Logger LOGGER = LoggerFactory.getLogger(MDCFilter.class);

	@Value("${logstash.trace.id.key:X-Trace-Id}")
	private String traceId;

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {
		LOGGER.debug("inicializando Filtro");
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
			throws IOException, ServletException {
		try {

			// Setup MDC data:
			// Referenced in Spring Boot's logging.pattern.level property
			MDC.put("clientIp", request.getRemoteAddr());
			if (HttpServletRequest.class.isAssignableFrom(request.getClass())) {
				MDC.put("requestId", requestId(((HttpServletRequest) request)));
				Enumeration<String> headers = ((HttpServletRequest) request).getHeaderNames();
				while (headers.hasMoreElements()) {
					String headerName = headers.nextElement();
					String headerValue = ((HttpServletRequest) request).getHeader(headerName);
					if (headerValue != null && headerName.toUpperCase().startsWith("X-")) {
						MDC.put(headerName, headerValue);
					}
				}
			}
			getProperties().keySet().forEach(property -> MDC.put(property, getProperties().get(property)));
			chain.doFilter(request, response);
		} finally {
			MDC.clear();
		}
	}

	@Override
	public void destroy() {
		MDC.clear();
	}

	protected String requestId(HttpServletRequest request) {
		//El api gateway tiene que agregar el Correlation Id
		String traceValue = request.getHeader(traceId);
		if (StringUtils.isEmpty(traceValue)) {
			LOGGER.warn("Request Id not present in request,one autogenerated, please configure your api gateway to generate trace-id");
			traceValue = UUID.randomUUID().toString();
		}
		return traceValue;
	}
}
