package com.crabshue.commons.http.client;

import com.crabshue.commons.http.client.client.ClientOptions;
import com.crabshue.commons.http.client.ssl.SslOptions;
import com.crabshue.commons.retryable.RetryOptions;
import lombok.NonNull;

/**
 * Builder for {@link HttpClient}.
 */
public class HttpClientBuilder {

    private ClientOptions clientOptions = ClientOptions.defaultClientOptions();

    private RetryOptions retryOptions = RetryOptions.defaultRetryOptions();

    private SslOptions sslOptions = SslOptions.defaultSslOptions();

    /**
     * Prepare a {@link HttpClient}.
     *
     * @return new instance.
     */
    public static HttpClientBuilder standard() {
        return new HttpClientBuilder();
    }

    /**
     * Set the client options.
     * <p>Default value is {@link ClientOptions#defaultClientOptions()}</p>
     *
     * @param clientOptions the client options.
     * @return instance.
     */
    public HttpClientBuilder withClientOptions(@NonNull final ClientOptions clientOptions) {

        this.clientOptions = clientOptions;
        return this;
    }

    /**
     * Set the retry options.
     * <p>Default value is {@link RetryOptions#defaultRetryOptions()}</p>
     *
     * @param retryOptions
     * @return instance.
     */
    public HttpClientBuilder withRetryOptions(@NonNull final RetryOptions retryOptions) {

        this.retryOptions = retryOptions;
        return this;
    }

    /**
     * @return
     */
    public HttpClientBuilder withSslOptions(@NonNull final SslOptions sslOptions) {

        this.sslOptions = sslOptions;
        return this;
    }

    /**
     * Build the {@link HttpClient}.
     *
     * @return the JAX-RS client.
     */
    public HttpClient build() {
        return new HttpClient(this.clientOptions, this.retryOptions, this.sslOptions);
    }


    private HttpClientBuilder() {
    }
}
