package com.crabshue.commons.http.client;

import java.util.Map;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.Response;

import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.http.client.client.ClientOptions;
import com.crabshue.commons.http.client.client.JerseyClientBuilder;
import com.crabshue.commons.http.client.exceptions.HttpClientErrorContext;
import com.crabshue.commons.http.client.exceptions.HttpClientErrorType;
import com.crabshue.commons.http.client.ssl.SslOptions;
import com.crabshue.commons.retryable.RetryOptions;
import com.crabshue.commons.retryable.RetryPolicyBuilder;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.RetryPolicy;

/**
 * JAX-RS client.
 *
 * @see HttpClientBuilder
 */
@Slf4j
public class HttpClient {

    private final ClientOptions clientOptions;

    private final RetryOptions retryOptions;

    private final SslOptions sslOptions;

    HttpClient(final ClientOptions clientOptions, final RetryOptions retryOptions, final SslOptions sslOptions) {
        this.clientOptions = clientOptions;
        this.retryOptions = retryOptions;
        this.sslOptions = sslOptions;
    }

    /**
     * Execute an {@link HttpRequest}
     *
     * @param httpRequest the HTTP request.
     * @return the {@link Response}.
     */
    public Response http(@NonNull final HttpRequest httpRequest) {

        final Invocation.Builder invocationBuilder = this.buildInvocation(httpRequest);

        final Entity entity = httpRequest.getEntity()
            .map(httpEntity -> Entity.entity(httpEntity, httpRequest.getEntityMediaType()))
            .orElse(null);

        final RetryPolicy retryPolicy = RetryPolicyBuilder.newRetryPolicy(this.retryOptions);

        // response status triggers retry callback
        retryPolicy.retryIf(response -> {
            if (retryOptions.getMaximumNumberOfRetries() > 0) {
                boolean willRetry = clientOptions.getStatusesForRetry().contains(((Response) response).getStatusInfo().toEnum());
                if (willRetry) {
                    logger.warn("Status response [{}] will trigger a retry because part of [{}]",
                        ((Response) response).getStatusInfo().toEnum(), clientOptions.getStatusesForRetry());
                }
                return willRetry;
            }
            return false;
        });

        return Failsafe.with(retryPolicy)
            .onFailedAttempt((o, throwable, executionContext) -> {
                if (retryOptions.getMaximumNumberOfRetries() > 0) {
                    logger.warn("Attempt #{} failed.", executionContext.getExecutions(), throwable);
                }
            })
            .get(() -> {
                    try {
                        Response ret = invocationBuilder.method(httpRequest.getHttpMethod(), entity);

                        logger.info("Request to [{}] [{}{}][{}] returned with status : [{} {}] ",
                            httpRequest.getHttpMethod(), httpRequest.getRequestBaseUrl(),
                            httpRequest.getResourcePath(), httpRequest.getQueryParameters(),
                            ret.getStatus(), ret.getStatusInfo());

                        return ret;
                    } catch (Exception e) {
                        throw new SystemException(HttpClientErrorType.HTTP_REQUEST_FAILED, e)
                            .addContextValue(HttpClientErrorContext.URL, httpRequest.getRequestBaseUrl() + httpRequest.getResourcePath())
                            .addContextValue(HttpClientErrorContext.QUERY_PARAMETERS, httpRequest.getQueryParameters());
                    }
                }
            );
    }

    private Invocation.Builder buildInvocation(@NonNull final HttpRequest httpRequest) {

        final Client client = JerseyClientBuilder.newClient(this.clientOptions, this.sslOptions);

        WebTarget webTarget = client
            .target(httpRequest.getRequestBaseUrl())
            .path(httpRequest.getResourcePath());

        for (Map.Entry<String, String> entry : httpRequest.getQueryParameters().entrySet()) {
            webTarget = webTarget.queryParam(entry.getKey(), entry.getValue());
        }

        Invocation.Builder invocationBuilder = webTarget.request(httpRequest.getMediaType());

        invocationBuilder.headers(httpRequest.getHeaders());

        httpRequest.getCookies().forEach(invocationBuilder::cookie);

        return invocationBuilder;
    }
}
