/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.kubernetes.client.dsl.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.fabric8.kubernetes.api.model.ListOptions;
import io.fabric8.kubernetes.api.model.Status;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.dsl.base.OperationSupport;
import io.fabric8.kubernetes.client.utils.Utils;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.WebSocket;
import okhttp3.WebSocketListener;
import okio.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RawWatchConnectionManager
implements Watch {
    private static final Logger logger = LoggerFactory.getLogger(RawWatchConnectionManager.class);
    private ObjectMapper objectMapper;
    private Watcher<String> watcher;
    private HttpUrl.Builder watchUrlBuilder;
    private final AtomicBoolean forceClosed = new AtomicBoolean();
    private final AtomicReference<String> resourceVersion;
    private final int reconnectLimit;
    private final int reconnectInterval;
    private int maxIntervalExponent;
    private final AtomicInteger currentReconnectAttempt = new AtomicInteger(0);
    private final AtomicReference<WebSocket> webSocketRef = new AtomicReference();
    private final ScheduledExecutorService executor;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final AtomicBoolean reconnectPending = new AtomicBoolean(false);
    private final ArrayBlockingQueue<Object> queue = new ArrayBlockingQueue(1);
    private WebSocket webSocket;
    private OkHttpClient clonedClient;

    public RawWatchConnectionManager(OkHttpClient okHttpClient, HttpUrl.Builder watchUrlBuilder, ListOptions listOptions, ObjectMapper objectMapper, Watcher<String> watcher, int reconnectLimit, int reconnectInterval, int maxIntervalExponent) {
        this.clonedClient = okHttpClient;
        this.watchUrlBuilder = watchUrlBuilder;
        this.objectMapper = objectMapper;
        this.watcher = watcher;
        this.reconnectLimit = reconnectLimit;
        this.reconnectInterval = reconnectInterval;
        this.resourceVersion = new AtomicReference<String>(listOptions.getResourceVersion());
        this.maxIntervalExponent = maxIntervalExponent;
        this.executor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread ret = new Thread(r, "Executor for Watch " + System.identityHashCode(this));
            ret.setDaemon(true);
            return ret;
        });
        this.runWatch();
    }

    private void runWatch() {
        if (this.resourceVersion.get() != null) {
            this.watchUrlBuilder.removeAllQueryParameters("resourceVersion");
            this.watchUrlBuilder.addQueryParameter("resourceVersion", this.resourceVersion.get());
        }
        HttpUrl watchUrl = this.watchUrlBuilder.build();
        String origin = watchUrl.url().getProtocol() + "://" + watchUrl.url().getHost();
        if (watchUrl.url().getPort() != -1) {
            origin = origin + ":" + watchUrl.url().getPort();
        }
        Request request = new Request.Builder().get().url(watchUrl).addHeader("Origin", origin).build();
        this.webSocket = this.clonedClient.newWebSocket(request, new WebSocketListener(){

            @Override
            public void onOpen(WebSocket webSocket, Response response) {
                logger.info("Websocket opened");
                RawWatchConnectionManager.this.webSocketRef.set(webSocket);
                RawWatchConnectionManager.this.currentReconnectAttempt.set(0);
                RawWatchConnectionManager.this.started.set(true);
                RawWatchConnectionManager.this.queue.clear();
                RawWatchConnectionManager.this.queue.add(true);
            }

            @Override
            public void onMessage(WebSocket webSocket, String text) {
                try {
                    Map watchEvent = RawWatchConnectionManager.this.objectMapper.readValue(text, HashMap.class);
                    String watchEventType = watchEvent.get("type").toString();
                    String watchObjectAsString = RawWatchConnectionManager.this.objectMapper.writeValueAsString(watchEvent.get("object"));
                    RawWatchConnectionManager.this.watcher.eventReceived(Watcher.Action.valueOf(watchEventType), watchObjectAsString);
                }
                catch (IOException exception) {
                    logger.error("Failed to deserialize watch response: " + exception.getMessage());
                }
            }

            @Override
            public void onMessage(WebSocket webSocket, ByteString bytes) {
                this.onMessage(webSocket, bytes.utf8());
            }

            @Override
            public void onClosing(WebSocket webSocket, int code, String reason) {
                logger.info("Socket closing: " + reason);
                webSocket.close(code, reason);
            }

            @Override
            public void onClosed(WebSocket webSocket, int code, String reason) {
                logger.debug("WebSocket close received. code: {}, reason: {}", (Object)code, (Object)reason);
                if (RawWatchConnectionManager.this.forceClosed.get()) {
                    logger.debug("Ignoring onClose for already closed/closing websocket");
                    return;
                }
                if (RawWatchConnectionManager.this.currentReconnectAttempt.get() >= RawWatchConnectionManager.this.reconnectLimit && RawWatchConnectionManager.this.reconnectLimit >= 0) {
                    RawWatchConnectionManager.this.closeEvent(new KubernetesClientException("Connection unexpectedly closed"));
                    return;
                }
                RawWatchConnectionManager.this.scheduleReconnect();
            }

            @Override
            public void onFailure(WebSocket webSocket, Throwable t, Response response) {
                if (RawWatchConnectionManager.this.forceClosed.get()) {
                    logger.debug("Ignoring onFailure for already closed/closing websocket", t);
                    if (response != null && response.body() != null) {
                        response.body().close();
                    }
                    return;
                }
                if (response != null && response.code() == 200) {
                    RawWatchConnectionManager.this.queue.clear();
                    RawWatchConnectionManager.this.queue.offer(new KubernetesClientException("Received 200 on websocket", response.code(), null));
                    response.body().close();
                    return;
                }
                if (response != null) {
                    Status status = OperationSupport.createStatus(response);
                    if (response.body() != null) {
                        response.body().close();
                    }
                    logger.warn("Exec Failure: HTTP {}, Status: {} - {}", response.code(), status.getCode(), status.getMessage(), t);
                    if (!RawWatchConnectionManager.this.started.get()) {
                        RawWatchConnectionManager.this.queue.clear();
                        RawWatchConnectionManager.this.queue.offer(new KubernetesClientException(status));
                    }
                } else {
                    logger.warn("Exec Failure", t);
                    if (!RawWatchConnectionManager.this.started.get()) {
                        RawWatchConnectionManager.this.queue.clear();
                        RawWatchConnectionManager.this.queue.offer(new KubernetesClientException("Failed to start websocket", t));
                    }
                }
                if (RawWatchConnectionManager.this.currentReconnectAttempt.get() >= RawWatchConnectionManager.this.reconnectLimit && RawWatchConnectionManager.this.reconnectLimit >= 0) {
                    RawWatchConnectionManager.this.closeEvent(new KubernetesClientException("Connection failure", t));
                    return;
                }
                RawWatchConnectionManager.this.scheduleReconnect();
            }
        });
    }

    private void scheduleReconnect() {
        logger.debug("Submitting reconnect task to the executor");
        if (!this.executor.isShutdown()) {
            this.executor.submit(new NamedRunnable("scheduleReconnect"){

                @Override
                public void execute() {
                    if (!RawWatchConnectionManager.this.reconnectPending.compareAndSet(false, true)) {
                        logger.debug("Reconnect already scheduled");
                        return;
                    }
                    RawWatchConnectionManager.this.webSocketRef.set(null);
                    try {
                        logger.debug("Scheduling reconnect task");
                        RawWatchConnectionManager.this.executor.schedule(new NamedRunnable("reconnectAttempt"){

                            @Override
                            public void execute() {
                                try {
                                    RawWatchConnectionManager.this.runWatch();
                                    RawWatchConnectionManager.this.reconnectPending.set(false);
                                }
                                catch (Exception e) {
                                    logger.error("Exception in reconnect", e);
                                    RawWatchConnectionManager.this.webSocketRef.set(null);
                                    RawWatchConnectionManager.this.closeEvent(new KubernetesClientException("Unhandled exception in reconnect attempt", e));
                                    RawWatchConnectionManager.this.close();
                                }
                            }
                        }, RawWatchConnectionManager.this.nextReconnectInterval(), TimeUnit.MILLISECONDS);
                    }
                    catch (RejectedExecutionException e) {
                        RawWatchConnectionManager.this.reconnectPending.set(false);
                    }
                }
            });
        }
    }

    public void waitUntilReady() {
        Utils.waitUntilReady(this.queue, 10L, TimeUnit.SECONDS);
    }

    @Override
    public void close() {
        logger.debug("Force closing the watch {}", (Object)this);
        this.closeEvent(null);
        this.closeWebSocket(this.webSocketRef.getAndSet(null));
        if (!this.executor.isShutdown()) {
            try {
                this.executor.shutdown();
                if (!this.executor.awaitTermination(1L, TimeUnit.SECONDS)) {
                    logger.warn("Executor didn't terminate in time after shutdown in close(), killing it in: {}", (Object)this);
                    this.executor.shutdownNow();
                }
            }
            catch (Throwable t) {
                throw KubernetesClientException.launderThrowable(t);
            }
        }
    }

    private void closeEvent(KubernetesClientException cause) {
        if (this.forceClosed.getAndSet(true)) {
            logger.debug("Ignoring duplicate firing of onClose event");
            return;
        }
        this.watcher.onClose(cause);
    }

    private void closeWebSocket(WebSocket ws) {
        if (ws != null) {
            logger.debug("Closing websocket {}", (Object)ws);
            try {
                if (!ws.close(1000, null)) {
                    logger.warn("Failed to close websocket");
                }
            }
            catch (IllegalStateException e) {
                logger.error("Called close on already closed websocket: {} {}", (Object)e.getClass(), (Object)e.getMessage());
            }
        }
    }

    private long nextReconnectInterval() {
        int exponentOfTwo = this.currentReconnectAttempt.getAndIncrement();
        if (exponentOfTwo > this.maxIntervalExponent) {
            exponentOfTwo = this.maxIntervalExponent;
        }
        long ret = this.reconnectInterval * (1 << exponentOfTwo);
        logger.debug("Current reconnect backoff is " + ret + " milliseconds (T" + exponentOfTwo + ")");
        return ret;
    }

    private static abstract class NamedRunnable
    implements Runnable {
        private final String name;

        public NamedRunnable(String name) {
            this.name = Objects.requireNonNull(name);
        }

        private void tryToSetName(String value) {
            try {
                Thread.currentThread().setName(value);
            }
            catch (SecurityException securityException) {
                // empty catch block
            }
        }

        @Override
        public final void run() {
            String oldName = Thread.currentThread().getName();
            this.tryToSetName(this.name + "|" + oldName);
            try {
                this.execute();
            }
            finally {
                this.tryToSetName(oldName);
            }
        }

        protected abstract void execute();
    }
}

