package com.github.coderahfei.esignspringbootstarter.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.coderahfei.esignspringbootstarter.config.EsignConfig;
import org.apache.tomcat.util.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.*;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.LinkedHashMap;

@Component
public class ESignHelper<T> {
    private static final Logger log = LoggerFactory.getLogger(ESignHelper.class);
    public static EsignConfig esignConfig;

    @Autowired
    public ESignHelper(EsignConfig esignConfig) {
        ESignHelper.esignConfig = esignConfig;
    }

    public static <T> T sendGet(Class<T> clazz, String accountsApi, String desc) throws JsonProcessingException {
        String result = getRequestHeader(null, accountsApi, "GET", desc);
        ObjectMapper objectMapper = new ObjectMapper();
        T t = objectMapper.readValue(result, clazz);
        return t;
    }

    public static <T> T sendDelete(Class<T> clazz, String accountsApi, String params, String desc) throws JsonProcessingException {
        String result = getRequestHeader(params, accountsApi, "DELETE", desc);
        ObjectMapper objectMapper = new ObjectMapper();
        T t = objectMapper.readValue(result, clazz);
        return t;
    }

    public static <T> T sendPut(Class<T> clazz, String accountsApi, String params, String desc) throws JsonProcessingException {
        String result = getRequestHeader(params, accountsApi, "PUT", desc);
        ObjectMapper objectMapper = new ObjectMapper();
        T t = objectMapper.readValue(result, clazz);
        return t;
    }

    public static <T> T sendPost(Class<T> clazz, String accountsApi, String params, String desc) throws JsonProcessingException {
        String result = getRequestHeader(params, accountsApi, "POST", desc);
        ObjectMapper objectMapper = new ObjectMapper();
        T t = objectMapper.readValue(result, clazz);
        return t;
    }

    /**
     * @param params      请求体
     * @param accountsApi 请求接口
     * @param method      请求方式
     * @param desc        请求方式
     * @return r
     */
    public static String getRequestHeader(String params, String accountsApi, String method, String desc) {
        log.info("esignConfig=【{}】", JackJsonUtils.toJson(esignConfig));
        // 个人创建账号接口请求地址
        String accountsApiUrl = esignConfig.getConfig().getDomainName() + accountsApi;

        String contentMD5;
        if ("POST".equals(method)) {
            // 对请求Body体内的数据计算ContentMD5
            try {
                contentMD5 = doContentMD5(params);
            } catch (Exception e) {
                throw new RuntimeException("Content-MD5加密失败," + e.getMessage());
            }

        } else if ("PUT".equals(method) && StringUtils.hasText(params)) {
            // 对请求Body体内的数据计算ContentMD5
            try {
                contentMD5 = doContentMD5(params);
            } catch (Exception e) {
                throw new RuntimeException("Content-MD5加密失败," + e.getMessage());
            }
        } else if ("DELETE".equals(method) && StringUtils.hasText(params)) {
            // 对请求Body体内的数据计算ContentMD5
            try {
                contentMD5 = doContentMD5(params);
            } catch (Exception e) {
                throw new RuntimeException("Content-MD5加密失败," + e.getMessage());
            }

        } else {
            // GET请求时ContentMD5为""
            contentMD5 = "{}";
        }
        // 构建待签名字符串
        String accept = "*/*";
        String contentType = "application/json; charset=UTF-8";
        String url = accountsApi;
        String date = "";
        String headers = "";

        StringBuffer sb = new StringBuffer();
        sb.append(method).append("\n").
                append(accept).append("\n").
                append(contentMD5).append("\n").
                append(contentType).append("\n").
                append(date).append("\n");

        if ("".equals(headers)) {
            sb.append(headers).append(url);
        } else {
            sb.append(headers).append("\n").append(url);
        }

        // 构建参与请求签名计算的明文
        String plaintext = sb.toString();
        // 计算请求签名值
        String reqSignature = "";
        try {
            reqSignature = doSignatureBase64(plaintext, esignConfig.getConfig().getAppKey());
        } catch (Exception e) {
            throw new RuntimeException("计算请求签名值失败" + e.getMessage());
        }
        // 获取时间戳(精确到毫秒)
        long timeStamp = timeStamp();

        // 构建请求头
        LinkedHashMap<String, String> header = new LinkedHashMap<String, String>();
        header.put("X-Tsign-Open-App-Id", esignConfig.getConfig().getAppId());
        header.put("X-Tsign-Open-Auth-Mode", "Signature");
        header.put("X-Tsign-Open-Ca-Timestamp", String.valueOf(timeStamp));
        header.put("Accept", accept);
        header.put("Content-Type", contentType);
        header.put("X-Tsign-Open-Ca-Signature", reqSignature);
        header.put("Content-MD5", contentMD5);

        String result = "";
        try {
            if ("POST".equals(method)) {
                // 发送POST请求
                log.info("POST请求" + desc + "接口,url为：{}，请求头为：{}，请求体为：{}", accountsApiUrl, header, params);
                result = HTTPHelper.sendPOST(accountsApiUrl, params, header, "UTF-8");
                log.info(desc + "，返回结果为：" + result);

            } else if ("GET".equals(method)) {
                // 发送GET请求
                log.info("GET请求" + desc + "接口,url为：{}，请求头为：{}", accountsApiUrl, header);
                result = HTTPHelper.sendGet(accountsApiUrl, header, "UTF-8");
                log.info(desc + "，返回结果为：" + result);
            } else if ("DELETE".equals(method)) {
                // 发送DELETE请求
                log.info("DELETE请求" + desc + "接口,url为：{}，请求头为：{}", accountsApiUrl, header);
                result = HTTPHelper.sendDelete(accountsApiUrl, header, "UTF-8");
                log.info(desc + "，返回结果为：" + result);
            } else if ("PUT".equals(method)) {
                log.info("PUT请求" + desc + "接口,url为：{}，请求头为：{}", accountsApiUrl, header);
                result = HTTPHelper.sendPut(accountsApiUrl, params, null, "UTF-8", header);
                log.info(desc + "，返回结果为：" + result);
            }
        } catch (Exception e) {
            throw new RuntimeException(desc + "接口请求失败" + e.getMessage());
        }

        return result;
    }

    /***
     * 计算字符串的Content-MD5
     * @param file 文件
     * @return r
     */
    public static String getStringContentMD5(File file) {
        // 获取文件MD5的二进制数组（128位）
        byte[] bytes = getFileMD5Bytes1282(file);
        // 对文件MD5的二进制数组进行base64编码
        return new String(Base64.encodeBase64(bytes));
    }

    /***
     * 获取文件MD5-二进制数组（128位）
     *
     * @param file f
     * @return r
     */
    public static byte[] getFileMD5Bytes1282(File file) {
        FileInputStream fis = null;
        byte[] md5Bytes = null;
        try {
//            File file = new File(filePath);
            fis = new FileInputStream(file);
            MessageDigest md5 = MessageDigest.getInstance("MD5");
            byte[] buffer = new byte[1024];
            int length = -1;
            while ((length = fis.read(buffer, 0, 1024)) != -1) {
                md5.update(buffer, 0, length);
            }
            md5Bytes = md5.digest();
            fis.close();
        } catch (FileNotFoundException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        } catch (NoSuchAlgorithmException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        } catch (IOException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }
        return md5Bytes;
    }


    /***
     *
     * @param str 待计算的消息
     * @return MD5计算后摘要值的Base64编码(ContentMD5)
     * @throws Exception 加密过程中的异常信息
     */
    public static String doContentMD5(String str) throws Exception {
        byte[] md5Bytes = null;
        MessageDigest md5 = null;
        String contentMD5 = null;
        try {
            md5 = MessageDigest.getInstance("MD5");
            // 计算md5函数
            md5.update(str.getBytes("UTF-8"));
            // 获取文件MD5的二进制数组（128位）
            md5Bytes = md5.digest();
            // 把MD5摘要后的二进制数组md5Bytes使用Base64进行编码（而不是对32位的16进制字符串进行编码）
            contentMD5 = new String(Base64.encodeBase64(md5Bytes), "UTF-8");
        } catch (NoSuchAlgorithmException e) {
            String msg = MessageFormat.format("不支持此算法: {0}", e.getMessage());
            Exception ex = new Exception(msg);
            ex.initCause(e);
            throw ex;
        } catch (UnsupportedEncodingException e) {
            String msg = MessageFormat.format("不支持的字符编码: {0}", e.getMessage());
            Exception ex = new Exception(msg);
            ex.initCause(e);
            throw ex;
        }
        return contentMD5;
    }

    /***
     * 计算请求签名值
     *
     * @param message 待计算的消息
     * @param secret 密钥
     * @return HmacSHA256计算后摘要值的Base64编码
     * @throws Exception 加密过程中的异常信息
     */
    public static String doSignatureBase64(String message, String secret) throws Exception {
        String algorithm = "HmacSHA256";
        Mac hmacSha256;
        String digestBase64 = null;
        try {
            hmacSha256 = Mac.getInstance(algorithm);
            byte[] keyBytes = secret.getBytes("UTF-8");
            byte[] messageBytes = message.getBytes("UTF-8");
            hmacSha256.init(new SecretKeySpec(keyBytes, 0, keyBytes.length, algorithm));
            // 使用HmacSHA256对二进制数据消息Bytes计算摘要
            byte[] digestBytes = hmacSha256.doFinal(messageBytes);
            // 把摘要后的结果digestBytes转换成十六进制的字符串
            // String digestBase64 = Hex.encodeHexString(digestBytes);
            // 把摘要后的结果digestBytes使用Base64进行编码
            digestBase64 = new String(Base64.encodeBase64(digestBytes), "UTF-8");
        } catch (NoSuchAlgorithmException e) {
            String msg = MessageFormat.format("不支持此算法: {0}", e.getMessage());
            Exception ex = new Exception(msg);
            ex.initCause(e);
            throw ex;
        } catch (UnsupportedEncodingException e) {
            String msg = MessageFormat.format("不支持的字符编码: {0}", e.getMessage());
            Exception ex = new Exception(msg);
            ex.initCause(e);
            throw ex;
        } catch (InvalidKeyException e) {
            String msg = MessageFormat.format("无效的密钥规范: {0}", e.getMessage());
            Exception ex = new Exception(msg);
            ex.initCause(e);
            throw ex;
        }
        return digestBase64;
    }

    /***
     * 获取时间戳(毫秒级)
     *
     * @return 毫秒级时间戳, 如 1578446909000
     */
    public static long timeStamp() {
        long timeStamp = System.currentTimeMillis();
        return timeStamp;
    }
}
