package com.gateway.connector.tcp.server;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.channel.ChannelHandlerContext;

public class SessionManager {
	  private final static Logger logger = LoggerFactory.getLogger(SessionManager.class);

	    protected List<SessionListener> sessionListeners = new ArrayList<SessionListener>();
	  
	    public void setSessionListeners(List<SessionListener> sessionListeners) {
	        this.sessionListeners = sessionListeners;
	    }

	    /**
	     * The set of currently active Sessions for this Manager, keyed by session
	     * identifier.
	     */
	    protected Map<String, Session> sessions = new ConcurrentHashMap<String, Session>();

	    public synchronized void addSession(Session session) {
	        if (null == session) {
	            return;
	        }
	        sessions.put(session.getSessionId(), session);
	        logger.debug("put a session " + session.getSessionId() + " to sessions!");
	    }

	    public synchronized void updateSession(String sessionId) {
	        Session session = sessions.get(sessionId);
	        session.setLastAccessedTime(System.currentTimeMillis());

	        sessions.put(sessionId, session);
	    }

	    /**
	     * Remove this Session from the active Sessions for this Manager.
	     */
	    public synchronized void removeSession(Session session) {
	        if (session == null) {
	            throw new IllegalArgumentException("session is null!");
	        }
	        removeSession(session.getSessionId());
	    }

	    public synchronized void removeSession(String sessionId) {
	        sessions.remove(sessionId);	    
	        logger.debug("remove the session " + sessionId + " from sessions!");
	    }

	    public Session getSession(String sessionId) {
	        return sessions.get(sessionId+"");
	    }

	    public Session[] getSessions() {
	        return sessions.values().toArray(new Session[0]);
	    }

	    public Set<String> getSessionKeys() {
	        return sessions.keySet();
	    }

	    public int getSessionCount() {
	        return sessions.size();
	    }

	    /**
	     * define timeout 5min
	     */
	    private int maxInactiveInterval = 5 * 60;

	    public int getMaxInactiveInterval() {
	        return maxInactiveInterval;
	    }

	    public void setMaxInactiveInterval(int maxInactiveInterval) {
	        this.maxInactiveInterval = maxInactiveInterval;
	    }
	 
	    public synchronized Session createSession(String sessionId,String userName, ChannelHandlerContext ctx) {
	        Session session = sessions.get(sessionId);
	        if (session != null) {
	            logger.debug("session " + sessionId + " exist!");
	            /**
	             * 如果在已经建立Connection(1)的Channel上，再建立Connection(2)
	             * session.close会将ctx关闭， Connection(2)和Connection(1)的Channel都将会关闭
	             * 断线之后再建立连接Connection(3)，由于Session是有一点延迟
	             * Connection(3)和Connection(1/2)的Channel不是同一个
	             * **/
	            // 如果session已经存在则销毁session
	            session.close();
	            logger.debug("session " + sessionId + " have been closed!");
	        }
	        logger.debug("create new session " + sessionId + ", ctx -> " + ctx.toString());

	        session = new Session();
	        session.setSessionId(sessionId);
	        session.setValid(true);
	        session.setMaxInactiveInterval(this.getMaxInactiveInterval());
	        session.setCreationTime(System.currentTimeMillis());
	        session.setLastAccessedTime(System.currentTimeMillis());
	        session.setSessionManager(this);
	        session.setUserName(userName);
	        session.setConnection(createTcpConnection(session, ctx));
	        logger.debug("create new session " + sessionId + " successful!");

	        for (SessionListener listener : sessionListeners) {
	            session.addSessionListener(listener);
	        }
	        logger.debug("add listeners to session " + sessionId + " successful! " + sessionListeners);

	        return session;
	    }

	    protected Connection createTcpConnection(Session session, ChannelHandlerContext ctx) {
	        Connection conn = new Connection(ctx);
	        conn.setConnectionId(session.getSessionId());
	        conn.setSession(session);
	        return conn;
	    }

}
