package com.gateway.connector.tcp.server;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

 
 

public class Session {
	/**
     * We are currently processing a session create, so bypass certain
     * IllegalStateException tests. NOTE: This value is not included in the
     * serialized version of this object.
     */
    protected transient volatile boolean connecting = false;

    /**
     * We are currently processing a session expiration, so bypass certain
     * IllegalStateException tests. NOTE: This value is not included in the
     * serialized version of this object.
     */
    protected transient volatile boolean closing = false;

    /**
     * The time this session was created, in milliseconds since midnight,
     * January 1, 1970 GMT.
     */
    protected long creationTime = 0L;

    /**
     * The last accessed time for this Session.
     */
    protected volatile long lastAccessedTime = creationTime;

    /**
     * Flag indicating whether this session is valid or not.
     */
    protected volatile boolean isValid = false;

    /**
     * The maximum time interval, in seconds, between client requests before the
     * container may invalidate this session. A negative time indicates that the
     * session should never time out.
     */
    protected int maxInactiveInterval = 5 * 60;

    /**
     * The collection of user data attributes associated with this Session.
     */
    protected Map<String, Object> attributes = new ConcurrentHashMap<String, Object>();

    public boolean isValid() {
        if (closing) {
            return true;
        }
        return (isValid);
    }

    public void setValid(boolean isValid) {
        this.isValid = isValid;
    }

    public void setCreationTime(long creationTime) {
        this.creationTime = creationTime;
    }

    public long getCreationTime() {
        return creationTime;
    }

    public void setLastAccessedTime(long lastAccessedTime) {
        this.lastAccessedTime = lastAccessedTime;
    }

    public long getLastAccessedTime() {
        return lastAccessedTime;
    }

    public void setMaxInactiveInterval(int maxInactiveInterval) {
        this.maxInactiveInterval = maxInactiveInterval;
    }

    public int getMaxInactiveInterval() {
        return maxInactiveInterval;
    }

    public void setAttribute(String name, Object value) {
        if (!isValid()) {
            throw new IllegalStateException("[setAttribute]Session already invalidated");
        }

        if (name == null)
            return;

        attributes.put(name, value);
    }

    public Object getAttribute(String name) {
        if (!isValid()) {
            throw new IllegalStateException("[getAttribute]Session already invalidated");
        }

        if (name == null)
            return null;

        return (attributes.get(name));
    }
    
    private final static Logger logger = LoggerFactory.getLogger(Session.class);

    /**
     * The session identifier of this Session.
     */
    private String sessionId = null;

    private transient List<SessionListener> listeners = new CopyOnWriteArrayList<SessionListener>();

    private transient Connection connection = null;

    /**
     * The Manager with which this Session is associated.
     */
    private transient SessionManager sessionManager = null;

    public void access() {
        // Check to see if access is in progress or has previously been called
        if (!isValid) {
            return;
        }
        lastAccessedTime = System.currentTimeMillis();
    }

    public void connect() {
        // Check to see if tellNew is in progress or has previously been called
        if (connecting || !isValid) {
            logger.debug("the session " + sessionId + " is connecting or isValid = false!");
            return;
        }
        connecting = true;
        connection.connect();
        addSessionEvent();

        connecting = false;
        logger.debug("the session " + sessionId + " is ready!");
    }
    private String userName;
    public String getUserName() {
		return userName;
	}

	public void setUserName(String userName) {
		this.userName = userName;
	}

	private void addSessionEvent() {
        SessionEvent event = new SessionEvent(this);
        for (SessionListener listener : listeners) {
            try {
                listener.sessionCreated(event);
                logger.debug("SessionListener " + listener + " .sessionCreated() is invoked successfully!");
            } catch (Exception e) {
                logger.error("addSessionEvent error.", e);
            }
        }
    }

    /**
     * Perform the internal processing required to invalidate this session,
     * without triggering an exception if the session has already expired.
     */
    public void close() {
        close(true);
    }

    /**
     * Perform the internal processing required to invalidate this session,
     * without triggering an exception if the session has already expired.
     *
     * @param notify Should we notify listeners about the demise of this session?
     */
    void close(boolean notify) {
        // Check to see if close is in progress or has previously been called
        if (closing || !isValid) {
            logger.debug("the session " + sessionId + " is closing or isValid = false!");
            return;
        }
        synchronized (this) {
            // Check again, now we are inside the sync so this code only runs
            // once
            // Double check locking - closing and isValid need to be volatile
            if (closing || !isValid) {
                logger.debug("the session " + sessionId + " is closing or isValid = false!");
                return;
            }
            // Mark this session as "being closed"
            closing = true;
            if (notify) {
                SessionEvent event = new SessionEvent(this);
                for (SessionListener listener : listeners) {
                    try {
                        listener.sessionDestroyed(event);
                        logger.debug("SessionListener " + listener + " .sessionDestroyed() is invoked successfully!");
                    } catch (Exception e) {
                        logger.error("sessionDestroyed error! " + e);
                    }
                }
            }
            setValid(false);

            connection.close();

            recycle();
            // We have completed close of this session
            closing = false;
            logger.debug("the session " + sessionId + " have been destroyed!");
        }
    }

    /**
     * Release all object references, and initialize instance variables, in
     * preparation for reuse of this object.
     */
    public void recycle() {
        logger.debug("the session " + sessionId + " is recycled!");
        // Remove this session from our manager's active sessions
        sessionManager.removeSession(this);

        // Reset the instance variables associated with this Session
        listeners.clear();
        listeners = null;
        creationTime = 0L;
        connecting = false;
        closing = false;
        sessionId = null;
        lastAccessedTime = 0L;
        maxInactiveInterval = -1;
        isValid = false;
        sessionManager = null;
    }

    public boolean expire() {
        //A negative time indicates that the session should never time out.
        if (maxInactiveInterval < 0)
            return false;

        long timeNow = System.currentTimeMillis();
        int timeIdle = (int) ((timeNow - lastAccessedTime) / 1000L);
        if (timeIdle >= maxInactiveInterval) {
            return true;
        }
        return false;
    }

    /**
     * Add a session event listener to this component.
     */
    public void addSessionListener(SessionListener listener) {
        if (null == listener) {
            throw new IllegalArgumentException("addSessionListener listener");
        }
        listeners.add(listener);
    }

    /**
     * Remove a session event listener from this component.
     */
    public void removeSessionListener(SessionListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("removeSessionListener listener");
        }
        listeners.remove(listener);
    }

    public void setSessionId(String sessionId) {
        this.sessionId = sessionId;
    }

    public String getSessionId() {
        return sessionId;
    }

    public void setSessionManager(SessionManager sessionManager) {
        this.sessionManager = sessionManager;
    }

    public SessionManager getSessionManager() {
        return sessionManager;
    }

    public void setConnection(Connection connection) {
        this.connection = connection;
    }

    public Connection getConnection() {
        return this.connection;
    }
}
