package com.gateway.connector.tcp.server;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.channel.ChannelHandlerContext;

public class ConnectorManager {
	private final static Logger logger = LoggerFactory.getLogger(ConnectorManager.class);

	private void send(SessionManager sessionManager, String sessionId, Object message) throws Exception {
		Session session = sessionManager.getSession(sessionId);
		if (session == null) {
			throw new Exception(String.format("session %s no exist.", sessionId));
		}
		try {
			session.getConnection().send(message);
		} catch (Exception e) {
			logger.error("ExchangeConnector send occur Exception.", e);
			session.close();
			throw e;
		}
	}

	public boolean isWritable(SessionManager sessionManager, String sessionId) throws Exception {
		Session session = sessionManager.getSession(sessionId);
		if (session == null) {
			throw new Exception(String.format("session %s no exist.", sessionId));
		}
		return session.getConnection().isWritable();
	}

	private SessionManager sessionManager = new SessionManager();

	public SessionManager getSessionManager() {
		return sessionManager;
	}

	public void send(String sessionId, Object message) throws Exception {
		send(sessionManager, sessionId, message);
	}

	public boolean isWritable(String sessionId) throws Exception {
		return isWritable(sessionManager, sessionId);
	}

	public boolean exist(String sessionId) throws Exception {
		Session session = sessionManager.getSession(sessionId);
		return session != null ? true : false;
	}

	private HeartbeatListener heartbeatListener = null;

	public ConnectorManager() {
		init();
	}

	public void init() {
		heartbeatListener = new HeartbeatListener(sessionManager);

		Thread heartbeatThread = new Thread(heartbeatListener, "heartbeatListener");
		heartbeatThread.setDaemon(true);
		heartbeatThread.start();
	}

	public void destroy() {
		heartbeatListener.stop();

		for (Session session : sessionManager.getSessions()) {
			session.close();
		}
		sessionManager = null;
		logger.info("TcpConnector destroy");
	}

	public void connect(ChannelHandlerContext ctx, MessageWrapper wrapper) {
		try {
			Session session = sessionManager.createSession(wrapper.getSessionId(), wrapper.getUserName(), ctx);
			session.addSessionListener(heartbeatListener);
			session.connect();

			sessionManager.addSession(session);
			/** send **/
			session.getConnection().send(wrapper.getBody());
		} catch (Exception e) {
			logger.error("Connector connect occur Exception.", e);
		}
	}

	public void close(MessageWrapper wrapper) {
		Session session = sessionManager.getSession(wrapper.getSessionId());
		if (session != null) {
			session.getConnection().send(wrapper.getBody());
			session.close();
		}
	}

	public void heartbeatClient(MessageWrapper wrapper) {
		try {
			sessionManager.updateSession(wrapper.getSessionId());
			Session session = sessionManager.getSession(wrapper.getSessionId());
			session.getConnection().send(wrapper.getBody());
		} catch (Exception e) {
			logger.error("TcpConnector heartbeatClient occur Exception.", e);
		}
	}

	public void responseSendMessage(MessageWrapper wrapper) {
		try {
			Session session = sessionManager.getSession(wrapper.getSessionId());
			if (session != null) {
				session.getConnection().send(wrapper.getBody());
			}
			else {
				logger.warn("{} session is not exist.", wrapper.getSessionId());
			}

		} catch (Exception e) {
			logger.error("TcpConnector responseSendMessage occur Exception.", e);
		}
	}

	public void responseNoKeepAliveMessage(ChannelHandlerContext ctx, MessageWrapper wrapper) {
		try {
			Connection connection = new Connection(ctx);
			connection.send(wrapper.getBody());
		} catch (Exception e) {
			logger.error("TcpConnector responseNoKeepAliveMessage occur Exception.", e);
		}
	}
}
