package com.github.bishoku.chunkprocessor;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import com.github.bishoku.chunkprocessor.ChunkProcessorHelper.ChunkProcessor;
import com.github.bishoku.chunkprocessor.ChunkProcessorHelper.RecordProcessor;
import com.github.bishoku.chunkprocessor.util.QueueUtil;

class Worker<T> implements Runnable {

  private final ChunkProcessor<T> chunkProcessor;
  private final RecordProcessor<T> recordProcessor;
  private final BlockingQueue<T> source;
  private final AtomicBoolean finished;
  private final AtomicBoolean forceToFinish;
  private final AtomicInteger processedRecordCount;
  private final int chunkSize;
  private final long chunkInterval;

  private Worker(ChunkProcessor<T> chunkProcessor, RecordProcessor<T> recordProcessor,
      BlockingQueue<T> source, AtomicBoolean finished, AtomicBoolean forceToFinish,
      AtomicInteger processedRecordCount, int chunkSize, long chunkInterval) {
    this.recordProcessor = recordProcessor;
    this.chunkProcessor = chunkProcessor;
    this.source = source;
    this.finished = finished;
    this.forceToFinish = forceToFinish;
    this.processedRecordCount = processedRecordCount;
    this.chunkInterval = chunkInterval;
    this.chunkSize = chunkSize;
  }

  public Worker(RecordProcessor<T> recordProcessor, BlockingQueue<T> source, AtomicBoolean finished,
      AtomicBoolean forceToFinish, AtomicInteger processedRecordCount, int chunkSize,
      long chunkInterval) {
    this(null, recordProcessor, source, finished, forceToFinish, processedRecordCount, chunkSize,
        chunkInterval);
  }

  public Worker(ChunkProcessor<T> chunkProcessor, BlockingQueue<T> source, AtomicBoolean finished,
      AtomicBoolean forceToFinish, AtomicInteger processedRecordCount, int chunkSize,
      long chunkInterval) {
    this(chunkProcessor, null, source, finished, forceToFinish, processedRecordCount, chunkSize,
        chunkInterval);
  }

  @Override
  public void run() {

    while (!Thread.currentThread().isInterrupted() && !forceToFinish.get()) {

      List<T> chunk = new ArrayList<>();
      try {
        int drainRecordCount =
            QueueUtil.drain(source, chunk, chunkSize, chunkInterval, TimeUnit.MILLISECONDS);
        if (drainRecordCount > 0 && !forceToFinish.get()) {
          if (null != chunkProcessor) {
            int processedCount = chunkProcessor.process(chunk);
            processedRecordCount.addAndGet(processedCount);
          } else if (null != recordProcessor) {
            for (int i = 0; i < drainRecordCount; i++) {
              recordProcessor.process(chunk.get(i));
              processedRecordCount.getAndIncrement();
            }
          }
        } else if (finished.get()) {
          System.out.println(
              String.format("Thread %s finished its work", Thread.currentThread().getName()));
          break;
        }
      } catch (InterruptedException e) {
        Thread.currentThread().interrupt();
      } catch (Exception e) {
        System.out.println("Worker encountered an exception. Stopp all processing");
        e.printStackTrace(System.out);
        forceToFinish.set(true);
      }
    }
  }
}
