package com.github.bishoku.chunkprocessor;

import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class ChunkProcessorHelper<T> {

  private CompletableFuture<Void>[] taskList;

  private final BlockingQueue<T> source = new LinkedBlockingQueue<>();
  private final AtomicBoolean finished = new AtomicBoolean(false);
  private final AtomicBoolean forceToFinish = new AtomicBoolean(false);
  private final AtomicBoolean started = new AtomicBoolean(false);
  private final AtomicInteger totalProcessedRecordCount = new AtomicInteger(0);

  private final int chunkSize;
  private final long chunkInterval;
  private final int workerCount;
  private final ChunkProcessor<T> chunkProcessor;
  private final RecordProcessor<T> recordProcessor;

  private ExecutorService executor;

  public static <T> ChunkProcessorHelper<T> newInstance(RecordProcessor<T> recordProcessor) {
    return new ChunkProcessorHelper<T>(recordProcessor);
  }

  public static <T> ChunkProcessorHelper<T> newInstance(ChunkProcessor<T> chunkProcessor) {
    return new ChunkProcessorHelper<T>(chunkProcessor);
  }

  public static <T> ChunkProcessorHelper<T> newInstance(int workerCount,
      RecordProcessor<T> recordProcessor) {
    return new ChunkProcessorHelper<T>(workerCount, recordProcessor);
  }

  public static <T> ChunkProcessorHelper<T> newInstance(int workerCount,
      ChunkProcessor<T> chunkProcessor) {
    return new ChunkProcessorHelper<T>(workerCount, chunkProcessor);
  }


  private ChunkProcessorHelper(RecordProcessor<T> recordProcessor) {
    this(Runtime.getRuntime().availableProcessors(), recordProcessor);
  }

  private ChunkProcessorHelper(ChunkProcessor<T> chunkProcessor) {
    this(Runtime.getRuntime().availableProcessors(), chunkProcessor);
  }

  private ChunkProcessorHelper(int workerCount, RecordProcessor<T> recordProcessor) {
    this(100, 500, workerCount, null, recordProcessor);
  }

  private ChunkProcessorHelper(int workerCount, ChunkProcessor<T> chunkProcessor) {
    this(100, 500, workerCount, chunkProcessor, null);
  }

  private ChunkProcessorHelper(int chunkSize, long chunkInterval, int workerCount,
      ChunkProcessor<T> chunkProcessor, RecordProcessor<T> recordProcessor) {
    super();
    this.chunkSize = chunkSize;
    this.chunkInterval = chunkInterval;
    this.workerCount = workerCount;
    this.chunkProcessor = chunkProcessor;
    this.recordProcessor = recordProcessor;
  }

  @SuppressWarnings("unchecked")
  public void start() {
    taskList = new CompletableFuture[workerCount];
    executor = Executors.newWorkStealingPool(workerCount);

    for (int i = 0; i < workerCount; i++) {
      Worker<T> worker = null;
      if (chunkProcessor != null) {
        worker = new Worker<T>(this.chunkProcessor, this.source, finished, forceToFinish,
            totalProcessedRecordCount, chunkSize, chunkInterval);
      } else if (recordProcessor != null) {
        worker = new Worker<T>(this.recordProcessor, this.source, finished, forceToFinish,
            totalProcessedRecordCount, chunkSize, chunkInterval);
      }
      if (null != worker) {
        CompletableFuture<Void> task = CompletableFuture.runAsync(worker);
        taskList[i] = task;
      }
    }
    this.started.set(true);
  }

  public void add(T record) {
    if (!this.started.get()) {
      throw new UnsupportedOperationException("start() method must be invoked first");
    }
    source.add(record);
  }

  public void add(List<T> records) {
    if (!this.started.get()) {
      throw new UnsupportedOperationException("start() method must be invoked first");
    }
    source.addAll(records);
  }

  public void finish() throws Exception {
    finished.set(true);
    CompletableFuture<Void> tasks = CompletableFuture.allOf(taskList);
    try {
      tasks.get();
    } catch (InterruptedException | ExecutionException e) {
      forceToFinish.set(true);
      throw e;
    } finally {
      executor.shutdown();
      executor.awaitTermination(chunkInterval * 2, TimeUnit.MILLISECONDS);
    }
  }

  public int getProcessedRecordCount() {
    return totalProcessedRecordCount.get();
  }

  @FunctionalInterface
  public interface ChunkProcessor<T> {
    public int process(List<T> t) throws Exception;
  }

  @FunctionalInterface
  public interface RecordProcessor<T> {
    public void process(T t) throws Exception;
  }
}
