/*
 * Copyright (C) 2011 Benny Bottema (benny@bennybottema.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.bbottema.javareflection.valueconverter.converters;

import lombok.experimental.UtilityClass;
import org.bbottema.javareflection.util.Function;
import org.bbottema.javareflection.util.Function.Functions;
import org.bbottema.javareflection.valueconverter.ValueFunction;
import org.bbottema.javareflection.valueconverter.ValueFunction.ValueFunctionImpl;
import org.jetbrains.annotations.Nullable;

import javax.activation.DataSource;
import javax.activation.FileDataSource;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Collection;

@Nullable
@UtilityClass
public final class FileConverters {
	
	public static final Collection<ValueFunction<File, ?>> FILE_CONVERTERS = produceFileConverters();
	
	private static Collection<ValueFunction<File, ?>> produceFileConverters() {
		ArrayList<ValueFunction<File, ?>> converters = new ArrayList<>();
		converters.add(new ValueFunctionImpl<>(File.class, File.class, Functions.<File>identity()));
		converters.add(new ValueFunctionImpl<>(File.class, DataSource.class, new FileToDataSourceFunction()));
		converters.add(new ValueFunctionImpl<>(File.class, byte[].class, new FileToByteArrayFunction()));
		converters.add(new ValueFunctionImpl<>(File.class, InputStream.class, new FileToInputStreamFunction()));
		return converters;
	}
	
	private static class FileToDataSourceFunction implements Function<File, DataSource> {
		@Override
		public DataSource apply(File file) {
			return new FileDataSource(file);
		}
	}
	
	private static class FileToByteArrayFunction implements Function<File, byte[]> {
		@Override
		public byte[] apply(File file) {
			try {
				return Files.readAllBytes(file.toPath());
			} catch (IOException e) {
				throw new RuntimeException("Was unable to read file content", e);
			}
		}
	}
	
	private static class FileToInputStreamFunction implements Function<File, InputStream> {
		@Override
		public InputStream apply(File file) {
			try {
				return new FileInputStream(file);
			} catch (FileNotFoundException e) {
				throw new AssertionError("File found, but also not found? Is this the real life...", e);
			}
		}
	}
}