package cn.org.atool.fluentmachine.persistence.demo;

import cn.org.atool.fluentmachine.context.Context;
import cn.org.atool.fluentmachine.context.EventContext;
import cn.org.atool.fluentmachine.persistence.ContextTestSaver;
import cn.org.atool.fluentmachine.saver.ContextSaver;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.junit.jupiter.api.Test;
import org.test4j.annotations.Mocks;
import org.test4j.junit5.Test4J;
import org.test4j.mock.Stubs;
import org.test4j.tools.commons.ResourceHelper;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import static cn.org.atool.fluentmachine.persistence.demo.CreateInstanceMachine.CreateEvent.*;
import static cn.org.atool.fluentmachine.persistence.demo.CreateInstanceMachine.CreateStatus.WAIT_BATCH_CREATE_AND_CALLBACK;


@Mocks(CreateMachineService.class)
public class CreateInstanceMachineTest extends Test4J {
    CreateInstanceMachineTestMocks mocks = new CreateInstanceMachineTestMocks();

    ScheduledExecutorService createService = new ScheduledThreadPoolExecutor(10,
        new BasicThreadFactory.Builder().namingPattern("done-created-%d").daemon(true).build());

    ScheduledExecutorService callbackService = new ScheduledThreadPoolExecutor(10,
        new BasicThreadFactory.Builder().namingPattern("done-callback-%d").daemon(true).build());

    @Test
    void printMachine() throws InterruptedException, ExecutionException {
        AtomicInteger created = new AtomicInteger(Integer.MAX_VALUE);
        AtomicInteger callback = new AtomicInteger(Integer.MAX_VALUE);
        int count = 10;
        String tradeNo = "xxx_123";

        ContextSaver saver = new ContextTestSaver();
        Stubs.nilFields(new CreateInstanceMachine()).init(saver);

        String file = System.getProperty("user.dir") + "/src/test/java/cn/org/atool/fluentmachine/persistence/demo/CreateSiMachine.puml";
        ResourceHelper.writeStringToFile(new File(file), CreateInstanceMachine.getMachine().plantUml());

        mocks.CreateMachineService.validateCreate$.restReturn(null);
        mocks.CreateMachineService.sendAsyncCreateInstanceMsg$.restReturn(null);
        mocks.CreateMachineService.createInstance$.just(1).thenAnswer(f -> {
            EventContext<CreatePara> ctx2 = f.arg(0);
            ctx2.getData().setSik("sik-xxxx");
            return null;
        });
        mocks.CreateMachineService.registerLifecycle$.just(1).restReturn(null);
        mocks.CreateMachineService.registerCapability$.just(1).restReturn(null);
        mocks.CreateMachineService.sendCapability$.just(1).restReturn(null);
        mocks.CreateMachineService.buildBatchJobs$.just(1).restReturn(null);
        mocks.CreateMachineService.sendAllBatchJobsMsg$.just(1).restAnswer(f -> {
            // 设置为10批次
            created.set(count);
            callback.set(count);
            return null;
        });
        mocks.CreateMachineService.isReservedMode$.restReturn(false);
        mocks.CreateMachineService.createByBatch$.restReturn(null);
        mocks.CreateMachineService.relateOrder$.restReturn(null);
        mocks.CreateMachineService.sendBatchCapability$.restAnswer(f -> {
            // 每处理一批, 计数加一
            System.out.println("+++++" + created.decrementAndGet());
            return null;
        });
        mocks.CreateMachineService.isFinishAllBatch$.restAnswer(f -> created.get() == 0);
        mocks.CreateMachineService.isReceivedAllProduceCallback$.restAnswer(f -> callback.get() == 0);
        mocks.CreateMachineService.doneProduceCallback$.restAnswer(f -> {
            // 每处理完一批, 计数减一
            System.out.println("======" + callback.decrementAndGet());
            return null;
        });
        mocks.CreateMachineService.callbackMilkWay$.restReturn(null);

        // 启动业务流程
        CreateInstanceMachine.getMachine().start(new Context<CreatePara>(tradeNo).setData(new CreatePara()));
        // createSI消息触发
        Object target = CreateInstanceMachine.getMachine().fire(tradeNo, ReceivedCreateMsg);
        want.string(String.valueOf(target)).eq(WAIT_BATCH_CREATE_AND_CALLBACK.name());

        List<Future> futures = new ArrayList<>();
        // 模拟接收到批次消息
        futures.addAll(this.mockDskBatchCreate(count, tradeNo, created::get));
        // 模拟产品线回调
        futures.addAll(this.mockProduceCallback(count, tradeNo, callback::get));
        // 待模拟线程全部结束
        this.waitFinishAllTasks(futures);

        Context<CreatePara> ctx = CreateInstanceMachine.getMachine().loadContext(tradeNo);

        want.string(ctx.getStateId()).eq("FINISH");
        want.string(ctx.getData().getSik()).eq("sik-xxxx");
    }

    /**
     * 待模拟线程全部结束
     *
     * @param futures
     * @throws InterruptedException
     * @throws java.util.concurrent.ExecutionException
     */
    private void waitFinishAllTasks(List<Future> futures) throws InterruptedException, java.util.concurrent.ExecutionException {
        for (Future future : futures) {
            System.out.println("Future:" + future.get());
        }
    }

    /**
     * 模拟接收到批次消息
     *
     * @param count
     * @param tradeNo
     * @param supplier
     * @return
     */
    private List<Future> mockDskBatchCreate(int count, String tradeNo, Supplier<Integer> supplier) {
        List<Future> futures = new ArrayList<>();
        for (int index = 0; index < count; index++) {
            Future future = createService.submit(() -> {
                CreateInstanceMachine.getMachine().fireByPretreatment(tradeNo, DoneOneBatchCreateJob, () -> "");
                return "created:" + supplier.get();
            });
            futures.add(future);
        }
        return futures;
    }

    /**
     * 模拟产品线回调
     *
     * @param count
     * @param tradeNo
     * @param supplier
     * @return
     */
    private List<Future> mockProduceCallback(int count, String tradeNo, Supplier<Integer> supplier) {
        List<Future> futures = new ArrayList<>();
        // 模拟产品线回调
        for (int index = 0; index < count; index++) {
            Future future = callbackService.submit(() -> {
                CreateInstanceMachine.getMachine().fireByPretreatment(tradeNo, DoneOneCallbackFromProduce, () -> "");
                return "callback:" + supplier.get();
            });
            futures.add(future);
        }
        return futures;
    }
}