package cn.org.atool.fluentmachine.persistence;

import cn.org.atool.fluentmachine.MachineFactory;
import cn.org.atool.fluentmachine.UnTypeMachine;
import cn.org.atool.fluentmachine.builder.UnTypeBuilder;
import cn.org.atool.fluentmachine.context.Context;
import cn.org.atool.fluentmachine.context.EventContext;
import cn.org.atool.fluentmachine.context.StatusMap;
import cn.org.atool.fluentmachine.context.TradeState;
import cn.org.atool.fluentmachine.interfaces.Action;
import cn.org.atool.fluentmachine.interfaces.MachineStatus;
import cn.org.atool.fluentmachine.persistence.db.ATM;
import cn.org.atool.fluentmachine.utils.PlantUml;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.test4j.hamcrest.matcher.modes.EqMode;
import org.test4j.junit5.Test4J;
import org.test4j.tools.datagen.DataMap;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;

@SuppressWarnings("rawtypes")
public class PersistenceTest extends Test4J implements Serializable {
    private static UnTypeMachine<Order> stateMachine;

    private static final String machineId = "simple-repository";

    private final ContextRepository repository = ContextRepository.repository("test", createDataSource("dataSource"));

    @BeforeAll
    public static void build() {
        UnTypeBuilder<Order> builder = MachineFactory.unType(machineId);
        builder.states(
            s -> s.route("STATE1", "STATE4"),
            s -> s.route("STATE2", "STATE4"),
            s -> s.route("STATE3", "STATE4")
        ).external(g -> g
            .from("STATE1", "STATE2", "STATE3")
            .to("STATE4")
            .on("EVENT1")
            .perform(PersistenceTest::doPerform)
        );

        stateMachine = builder.create();
        String uml = PlantUml.activity(stateMachine);
        System.out.println(uml);
    }

    @Test
    public void testSave() {
        db.cleanTable(ATM.table.fluentMachineCtx, ATM.table.fluentMachineLog);
        // 保证业务单号存在
        ATM.dataMap.fluentMachineCtx.table(1)
            .machineId.values(machineId)
            .tradeNo.values("o-123")
            .env.values("test")
            .cleanAndInsert();
        ((MachineStatus) stateMachine).setSaver(repository);
        Context<Order> ctx = new Context<>(machineId, "o-123", new Order("o-123", 345));
        ctx.setSwitcher("kkk", true);
        ctx.setError("ekey", "adfasd");
        String target = (String) stateMachine.fire("STATE2", ctx, "EVENT1");
        want.string(target).eq("STATE4");
        db.table(ATM.table.fluentMachineCtx).query().eqDataMap(ATM.dataMap.fluentMachineCtx.table(1)
            .machineId.values(machineId)
            .tradeNo.values("o-123")
            .ctxState.values("STATE4")
            .regionStates.values("{\"STATE4\":{}}")
            .context.values("{\"orderId\":\"o-123\",\"userId\":345}")
            .actionErrors.values("{\"ekey\":\"adfasd\"}")
            .switcher.values("{\"kkk\":\"true\"}")
        );
        db.table(ATM.table.fluentMachineLog).query().eqDataMap(ATM.dataMap.fluentMachineLog.table(1)
            .machineId.values("simple-repository")
            .tradeNo.values("o-123")
            .ctxState.values("STATE4")
            .fireEvent.values("EVENT1")
            .actionErrors.values("{\"ekey\":\"adfasd\"}")
            .switcher.values("{\"kkk\":\"true\"}")
        );
    }

    @Test
    void loadContext() {
        db.cleanTable(ATM.table.fluentMachineCtx, ATM.table.fluentMachineLog);
        db.table(ATM.table.fluentMachineCtx).insert(ATM.dataMap.fluentMachineCtx.table(1)
            .machineId.values(machineId)
            .tradeNo.values("o-123")
            .ctxState.values("STATE4")
            .regionStates.values("{\"STATE4\":{}}")
            .context.values("{\"orderId\":\"o-123\",\"userId\":345}")
            .actionErrors.values("{\"ekey\":\"adfasd\"}")
            .switcher.values("{\"kkk\":\"true\"}")
            .env.values("test")
        );
        Context<Order> ctx = repository.loadContext(machineId, "o-123", true, Order.class);
        Context<Order> expected = new Context<Order>()
            .setTradeNo("o-123")
            .setData(new Order().setOrderId("o-123").setUserId(345L))
            .setError("ekey", "adfasd")
            .setSwitcher("kkk", true);

        expected
            .setStateId("STATE4")
            .setStates(new StatusMap().set("STATE4", Collections.emptyMap()));
        want.object(ctx).notNull().eqByProperties(new String[]{"tradeNo", "data", "stateId", "states"},
            expected, EqMode.IGNORE_DEFAULTS);
    }

    @Test
    void findTradesByStatus() {
        db.cleanTable(ATM.table.fluentMachineCtx, ATM.table.fluentMachineLog);
        db.table(ATM.table.fluentMachineCtx).insert(ATM.dataMap.fluentMachineCtx.table(3)
            .machineId.values(machineId)
            .tradeNo.values("o-123", "o-124", "o-125")
            .ctxState.values("STATE4")
            .env.values("test")
        );
        List<TradeState> list = repository.findTradesByStatus(machineId, "STATE4", 0, 100);
        want.list(list).sizeEq(3).eqByProperties("tradeNo", new String[]{"o-123", "o-124", "o-125"});
    }

    @Test
    void findTimeoutTradeNo() {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.MONDAY, -1);
        db.cleanTable(ATM.table.fluentMachineCtx, ATM.table.fluentMachineLog);
        db.table(ATM.table.fluentMachineCtx).insert(ATM.dataMap.fluentMachineCtx.table(3)
            .machineId.values(machineId)
            .tradeNo.values("o-123", "o-124", "o-125")
            .ctxState.values("STATE4")
            .env.values("test")
            .gmtModified.values(cal.getTime())
        );
        List<TradeState> states = repository.findTimeoutTradeNo(machineId, 60, 0, true, null, 0, 100);
        want.list(states).sizeEq(3).eqMap(DataMap.create(3)
            .kv("tradeNo", "o-123", "o-124", "o-125")
            .kv("stateId", "STATE4")
        );
        db.sqlList().wantFirstSql().end("" +
            "is_deleted=0 AND machine_id=? AND env=? " +
            "AND gmt_modified BETWEEN DATE_ADD(NOW(), INTERVAL ? SECOND) AND DATE_ADD(NOW(), INTERVAL ? SECOND) " +
            "AND id > ? ORDER BY id LIMIT ?");
        //db.sqlList().wantFirstPara().eqList(machineId, "test", -60, 0, 100);

        repository.findTimeoutTradeNo(machineId, 5, 0, true, new String[]{"STATE1", "STATE2"}, 0, 100);
        db.sqlList().wantSql(1).end("" +
            "is_deleted=0 AND machine_id=? AND env=? " +
            "AND ctx_state IN ('STATE1', 'STATE2') " +
            "AND gmt_modified BETWEEN DATE_ADD(NOW(), INTERVAL ? SECOND) AND DATE_ADD(NOW(), INTERVAL ? SECOND) " +
            "AND id > ? ORDER BY id LIMIT ?");

        repository.findTimeoutTradeNo(machineId, 5, 0, false, new String[]{"STATE1", "STATE2"}, 0, 100);
        db.sqlList().wantSql(2).end("" +
            "is_deleted=0 AND machine_id=? AND env=? " +
            "AND ctx_state NOT IN ('STATE1', 'STATE2') " +
            "AND gmt_modified BETWEEN DATE_ADD(NOW(), INTERVAL ? SECOND) AND DATE_ADD(NOW(), INTERVAL ? SECOND) " +
            "AND id > ? ORDER BY id LIMIT ?");
    }

    @Action("do perform")
    public static void doPerform(EventContext<Order> stringEventContext) {
    }
}