package cn.org.atool.fluentmachine.persistence;

import cn.org.atool.fluentmachine.state.IName;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

import static java.util.stream.Collectors.joining;

/**
 * 状态机上下文操作SQL语句
 *
 * @author darui.wu
 */
public interface ContextRepositorySql {
    /**
     * id
     */
    String F_ID = "id";
    /**
     * 状态机标识
     */
    String F_MACHINE_ID = "machine_id";
    /**
     * 业务单号
     */
    String F_TRADE_NO = "trade_no";
    /**
     * 当前主节点
     */
    String F_CTX_STATE = "ctx_state";
    /**
     * 版本锁
     */
    String F_LOCK_VERSION = "lock_version";
    /**
     * 锁过期时间
     */
    String F_LOCK_EXPIRE_TIME = "lock_expire_time";
    /**
     * 流程节点
     */
    String F_REGION_STATES = "region_states";
    /**
     * 上下文
     */
    String F_CONTEXT = "context";
    /**
     * 过程中错误信息
     */
    String F_ERRORS = "action_errors";
    /**
     * 过程中控制开关点
     */
    String F_SWITCHER = "switcher";
    /**
     * 触发事件
     */
    String F_FIRE_EVENT = "fire_event";
    /**
     * 来源节点
     */
    String F_SOURCE_STATE = "source_state";
    /**
     * 目标节点
     */
    String F_TARGET_STATE = "target_state";
    /**
     * 部署环境
     */
    String F_ENV = "env";
    /**
     * 逻辑删除标识
     */
    String F_IS_DELETED = "is_deleted";
    /**
     * 最后修改时间
     */
    String F_GMT_MODIFIED = "gmt_modified";
    /**
     * 记录创建时间
     */
    String F_GMT_CREATE = "gmt_create";

    /**
     * ctx全字段列表
     */
    List<String> CTX_ALL_FIELDS = Arrays.asList(
        F_MACHINE_ID, F_TRADE_NO,
        F_CTX_STATE, F_REGION_STATES, F_CONTEXT, F_ERRORS, F_SWITCHER,
        F_LOCK_VERSION, F_ENV);
    /**
     * log全字段列表
     */
    List<String> LOG_ALL_FIELDS = Arrays.asList(
        F_MACHINE_ID, F_TRADE_NO,
        F_CTX_STATE, F_REGION_STATES, F_CONTEXT, F_ERRORS, F_SWITCHER,
        F_FIRE_EVENT, F_SOURCE_STATE, F_TARGET_STATE, F_ENV);

    /**
     * 更新上下文
     */
    static String SQL_UPDATE_CONTEXT_STATUS(String ctxTable) {
        return " UPDATE " + ctxTable + " SET " +
            F_CTX_STATE + "=?, " +
            F_REGION_STATES + "=?, " +
            F_CONTEXT + "=?, " +
            F_ERRORS + "=?, " +
            F_SWITCHER + "=?, " +
            F_LOCK_EXPIRE_TIME + "=DATE_ADD(now(), INTERVAL ? SECOND), " +
            F_GMT_MODIFIED + "=now()" +
            " WHERE " + F_TRADE_NO + "=?" +
            " AND " + F_MACHINE_ID + "=?" +
            " AND " + F_ENV + "=?";
    }

    /**
     * 尝试获取指定上下文操作的锁
     */
    static String SQL_UPDATE_CONTEXT_LOCK(String ctxTable) {
        return "UPDATE " + ctxTable + " SET " +
            F_LOCK_VERSION + "=?, " +
            F_LOCK_EXPIRE_TIME + "=DATE_ADD(now(), INTERVAL ? SECOND), " +
            F_GMT_MODIFIED + "=now() " +
            " WHERE " + F_TRADE_NO + "=?" +
            " AND " + F_MACHINE_ID + "=?" +
            " AND " + F_ENV + "=? " +
            " AND (" + F_LOCK_EXPIRE_TIME + " < now() OR " + F_LOCK_VERSION + "=? OR " + F_LOCK_VERSION + "='0')";
    }

    /**
     * 新增上下文
     */
    static String SQL_INSERT_CONTEXT_STATUS(String ctxTable) {
        return "INSERT INTO " + ctxTable + " (" +
            String.join(", ", CTX_ALL_FIELDS) + ", " +
            F_LOCK_EXPIRE_TIME + ", " +
            F_IS_DELETED + ", " +
            F_GMT_CREATE + ", " +
            F_GMT_MODIFIED + ")" +
            " VALUES(" +
            CTX_ALL_FIELDS.stream().map(f -> ("?")).collect(joining(", ")) + ", " +
            " DATE_ADD(now(), INTERVAL ? SECOND), " +
            " 0, " +
            " now(), " +
            " now()" +
            ")";
    }

    /**
     * 新增上下文变更log
     */
    static String SQL_INSERT_CONTEXT_LOG(String logTable) {
        return "INSERT INTO " + logTable + " (" +
            String.join(", ", LOG_ALL_FIELDS) + ", " +
            F_IS_DELETED + ", " +
            F_GMT_CREATE + ", " +
            F_GMT_MODIFIED + ") VALUES (" +
            LOG_ALL_FIELDS.stream().map(f -> ("?")).collect(joining(", ")) + ", " +
            " 0, " +
            " now(), " +
            " now())";
    }

    /**
     * 判断是否存在对应tradeNo的上下文
     */
    static String SQL_IS_EXISTS_CONTEXT(String ctxTable) {
        return "SELECT 1 FROM " + ctxTable +
            " WHERE " + F_MACHINE_ID + "=?" +
            " AND " + F_TRADE_NO + "=?" +
            " AND " + F_ENV + "=? " +
            " AND " + F_IS_DELETED + "=0";
    }

    /**
     * 根据业务号查找对应的上下文
     */
    static String SQL_FIND_CONTEXT_BY_TRADE_NO(String ctxTable, boolean ignoreLock) {
        String sql = "SELECT " + String.join(", ", CTX_ALL_FIELDS) +
            " FROM  " + ctxTable +
            " WHERE " + F_IS_DELETED + "=0 " +
            " AND " + F_MACHINE_ID + "=? " +
            " AND " + F_TRADE_NO + "=? " +
            " AND " + F_ENV + "=? ";
        if (ignoreLock) {
            return sql;
        } else {
            return sql + " AND (" + F_LOCK_EXPIRE_TIME + " < now() OR " + F_LOCK_VERSION + "='0')";
        }
    }

    String TRADE_STATUS_FIELDS = String.join(", ",
        F_ID, F_MACHINE_ID, F_TRADE_NO, F_CTX_STATE, F_REGION_STATES, F_GMT_CREATE, F_GMT_MODIFIED);

    /**
     * 按主节点批量查询上下文
     */
    static String SQL_FIND_CONTEXT_BY_STATUS(String ctxTable) {
        return "SELECT " + TRADE_STATUS_FIELDS +
            " FROM " + ctxTable +
            " WHERE " + F_IS_DELETED + "=0" +
            " AND " + F_MACHINE_ID + "=?" +
            " AND " + F_ENV + "=?" +
            " AND " + F_CTX_STATE + "=?" +
            " AND id > ? " +
            " ORDER BY id LIMIT ?";
    }

    /**
     * 查找超时状态机流程
     *
     * @return sql
     */
    static String SQL_FIND_CONTEXT_TIMEOUT(String ctxTable, boolean isIn, Object[] states) {
        String inOrNotStr = states == null || states.length == 0 ? "" :
            " AND " + F_CTX_STATE + (isIn ? " IN" : " NOT IN") + " (" + joinStates(states) + ")";

        return "SELECT " + TRADE_STATUS_FIELDS +
            " FROM " + ctxTable +
            " WHERE " + F_IS_DELETED + "=0" +
            " AND " + F_MACHINE_ID + "=?" +
            " AND " + F_ENV + "=?" + inOrNotStr +
            " AND " + F_GMT_MODIFIED + " BETWEEN DATE_ADD(NOW(), INTERVAL ? SECOND) AND DATE_ADD(NOW(), INTERVAL ? SECOND)" +
            " AND id > ? " +
            " ORDER BY id LIMIT ?";
    }

    static String joinStates(Object[] states) {
        return Stream.of(states)
            .map(IName::name)
            .collect(joining("', '", "'", "'"));
    }
}