package cn.org.atool.fluentmachine.persistence;

import cn.org.atool.fluentmachine.context.Context;
import cn.org.atool.fluentmachine.context.StatusMap;

import java.util.HashMap;
import java.util.Map;

import static cn.org.atool.fluentmachine.persistence.ContextRepositorySql.*;
import static cn.org.atool.fluentmachine.persistence.JsonKit.kit;

/**
 * ContextHelper: 上下文序列化帮助类
 *
 * @author darui.wu
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public class ContextHelper {
    /**
     * 数据库结构转换为Context值
     *
     * @param map   记录行
     * @param klass Context泛型对象
     * @return Context
     */
    public static <DATA> Context<DATA> toContext(Map<String, Object> map, Class<DATA> klass) {
        Context<DATA> ctx = new Context();

        ctx.setMachineId((String) map.get(F_MACHINE_ID));
        ctx.setTradeNo((String) map.get(F_TRADE_NO));
        ctx.setStateId((String) map.get(F_CTX_STATE));

        StatusMap status = kit.fromJson((String) map.get(F_REGION_STATES), StatusMap.class);
        ctx.setStates(status);

        DATA data = kit.fromJson((String) map.get(F_CONTEXT), klass);
        ctx.setData(data);
        Map<String, String> errors = kit.fromJson((String) map.get(F_ERRORS), HashMap.class);
        ctx.setErrors(errors);
        Map<String, String> switcher = kit.fromJson((String) map.get(F_SWITCHER), HashMap.class);
        ctx.setSwitcher(switcher);

        return ctx;
    }

    /**
     * 通过序列化和反序列化进行上下文深度拷贝
     *
     * @param source Context
     * @return copy Context
     */
    public static <DATA> Context<DATA> copy(Context<DATA> source) {
        Context<DATA> target = new Context<>();

        target.setMachineId(source.getMachineId());
        target.setTradeNo(source.getTradeNo());
        target.setStateId(source.getStateId());

        target.setStates(kit.fromJson(kit.toJson(source.getStates()), StatusMap.class));
        if (source.getData() == null) {
            target.setData(null);
        } else {
            target.setData((DATA) kit.fromJson(kit.toJson(source.getData()), source.getData().getClass()));
        }
        target.setErrors(kit.fromJson(kit.toJson(source.getErrors()), HashMap.class));
        target.setSwitcher(kit.fromJson(kit.toJson(source.getSwitcher()), HashMap.class));
        return target;
    }

    /**
     * 上下文数据转换为数据库模型
     *
     * @param ctx Context
     * @return ContextEntity
     */
    public static ContextEntity toSaveContext(Context ctx) {
        ContextEntity bean = new ContextEntity();

        bean.setMachineId(ctx.getMachineId());
        bean.setTradeNo(ctx.getTradeNo());
        bean.setCtxState(ctx.getStateId());
        bean.setRegionStates(kit.toJson(ctx.getStates()));
        bean.setContext(kit.toJson(ctx.getData()));
        bean.setErrors(kit.toJson(ctx.getErrors()));
        bean.setSwitcher(kit.toJson(ctx.getSwitcher()));
        return bean;
    }
}