package cn.org.atool.fluentmachine.persistence;

import cn.org.atool.fluentmachine.context.Context;
import cn.org.atool.fluentmachine.context.FireContext;
import cn.org.atool.fluentmachine.context.StatusMap;
import cn.org.atool.fluentmachine.context.TradeState;
import cn.org.atool.fluentmachine.exception.LockException;
import lombok.Getter;
import lombok.Setter;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import javax.sql.DataSource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import static cn.org.atool.fluentmachine.persistence.ContextRepositorySql.*;
import static cn.org.atool.fluentmachine.persistence.JsonKit.kit;
import static cn.org.atool.fluentmachine.utils.Utils.assertNotNull;
import static java.lang.Math.abs;
import static java.lang.Math.max;
import static java.util.stream.Collectors.toList;

/**
 * 保存状态机上下文数据库实现
 *
 * @author darui.wu
 */
@SuppressWarnings({"rawtypes", "UnusedReturnValue", "unused"})
public class ContextRepositoryImpl extends JdbcDaoSupport implements ContextRepository {
    /**
     * 默认超时时间30s
     */
    @Getter
    @Setter
    private long lockDuration = 30;

    private final String env;

    private final String ctxTable;

    private final String logTable;

    public ContextRepositoryImpl(String env, DataSource dataSource, String ctxTable, String logTable) {
        super.setDataSource(dataSource);
        this.env = env;
        this.ctxTable = ctxTable;
        this.logTable = logTable;
    }

    @Override
    public void saveContext(Context ctx, FireContext fire, boolean isNew) {
        assertNotNull("JdbcTemplate", super.getJdbcTemplate());
        ContextEntity bean = ContextHelper.toSaveContext(ctx);
        if (!isNew) {
            super.getJdbcTemplate().update(
                SQL_UPDATE_CONTEXT_STATUS(this.ctxTable),
                bean.getCtxState(), bean.getRegionStates(), bean.getContext(), bean.getErrors(), bean.getSwitcher(),
                this.lockDuration,
                ctx.getTradeNo(), ctx.getMachineId(), this.env);
        } else {
            super.getJdbcTemplate().update(
                SQL_INSERT_CONTEXT_STATUS(this.ctxTable),
                ctx.getMachineId(), ctx.getTradeNo(),
                bean.getCtxState(), bean.getRegionStates(), bean.getContext(), bean.getErrors(), bean.getSwitcher(),
                ctx.getLockVersion(), this.env, ctx.getLockExpireSecond());
        }
        super.getJdbcTemplate().update(SQL_INSERT_CONTEXT_LOG(this.logTable),
            ctx.getMachineId(), ctx.getTradeNo(),
            bean.getCtxState(), bean.getRegionStates(), bean.getContext(), bean.getErrors(), bean.getSwitcher(),
            fire.getFireEvent(), fire.getSourceState(), fire.getTargetState(), this.env);
    }

    @Override
    public boolean lock(Context ctx, String lockVersion, Object event) {
        boolean exists = this.isExistContext(ctx.getMachineId(), ctx.getTradeNo());
        if (exists) {
            return updateLock(ctx, lockVersion, this.lockDuration);
        } else {
            return true;
        }
    }

    /**
     * 一年时间 365 天(24 小时(60 分钟(60秒)))
     */
    private static final long aYear = 365 * 24 * 60 * 60;

    @Override
    public boolean unlock(Context ctx) {
        return updateLock(ctx, "0", -aYear);
    }

    /**
     * 更新全局锁
     *
     * @param ctx           上下文
     * @param newLock       新的版本锁
     * @param expireSeconds 版本锁过期时间
     * @return true: lock success
     */
    private boolean updateLock(Context ctx, String newLock, long expireSeconds) {
        assertNotNull("JdbcTemplate", super.getJdbcTemplate());
        int result = super.getJdbcTemplate().update(
            SQL_UPDATE_CONTEXT_LOCK(this.ctxTable),
            newLock, expireSeconds,
            ctx.getTradeNo(), ctx.getMachineId(), this.env, ctx.getLockVersion());
        if (result > 0) {
            ctx.setLockVersion(newLock);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isExistContext(String machineId, String tradeNo) {
        assertNotNull("JdbcTemplate", super.getJdbcTemplate());
        List<Map<String, Object>> list = super.getJdbcTemplate()
            .queryForList(SQL_IS_EXISTS_CONTEXT(this.ctxTable), machineId, tradeNo, this.env);
        return list.size() > 0;
    }

    @Override
    public <DATA> Context<DATA> loadContext(String machineId, String tradeNo, boolean ignoreLock, Class<DATA> klass) {
        if (klass == null || Objects.equals(Object.class, klass)) {
            throw new RuntimeException("The context type is not specified.");
        }
        assertNotNull("JdbcTemplate", super.getJdbcTemplate());
        List<Map<String, Object>> list = super.getJdbcTemplate().queryForList(
            SQL_FIND_CONTEXT_BY_TRADE_NO(this.ctxTable, ignoreLock), machineId, tradeNo, this.env);
        Context<DATA> ctx = list.size() == 0 ? null : ContextHelper.toContext(list.get(0), klass);
        if (ctx == null) {
            throw new LockException("The context[ignoreLock=%s, machineId=%s, tradeNo=%s, env=%s] not found or is locked.",
                ignoreLock, machineId, tradeNo, this.env);
        } else {
            return ctx;
        }
    }

    @Override
    public List<TradeState> findTradesByStatus(String machineId, String stateId, long id, int limit) {
        assertNotNull("JdbcTemplate", super.getJdbcTemplate());
        List<Map<String, Object>> list = super.getJdbcTemplate().queryForList(
            SQL_FIND_CONTEXT_BY_STATUS(this.ctxTable),
            machineId, this.env, stateId, id, limit);
        return list.stream().map(this::toTradeStatus).collect(toList());
    }

    @Override
    public List<TradeState> findTimeoutTradeNo(String machineId, long minTimeout, long maxTimeout, boolean isIn, Object[] states, long id, int limit) {
        assertNotNull("JdbcTemplate", super.getJdbcTemplate());
        long min = max(10, abs(minTimeout)) * -1;
        long max = max(60, maxTimeout <= 0 ? aYear : maxTimeout) * -1;
        if (max >= min) {
            max = min - 60;
        }
        List<Map<String, Object>> list = super.getJdbcTemplate().queryForList(
            SQL_FIND_CONTEXT_TIMEOUT(this.ctxTable, isIn, states),
            machineId, this.env, max, min, id, limit);
        return list.stream().map(this::toTradeStatus).collect(toList());
    }

    private TradeState toTradeStatus(Map<String, Object> m) {
        TradeState c = new TradeState();
        c.setId(((Number) m.get(F_ID)).longValue());
        c.setMachineId((String) m.get(F_MACHINE_ID));
        c.setTradeNo((String) m.get(F_TRADE_NO));
        c.setStateId((String) m.get(F_CTX_STATE));
        c.setStates(kit.fromJson((String) m.get(F_REGION_STATES), StatusMap.class));
        c.setGmtCreated((Date) m.get(F_GMT_CREATE));
        c.setGmtModified((Date) m.get(F_GMT_MODIFIED));
        return c;
    }
}