package cn.org.atool.fluentmachine.persistence;

import cn.org.atool.fluentmachine.context.Context;
import cn.org.atool.fluentmachine.context.FireContext;
import org.springframework.jdbc.core.support.JdbcDaoSupport;

import javax.sql.DataSource;
import java.util.List;
import java.util.Map;

import static cn.org.atool.fluentmachine.persistence.ContextRepositorySql.*;
import static java.util.stream.Collectors.toMap;

/**
 * 保存状态机上下文数据库实现
 *
 * @author darui.wu
 */
public class ContextRepositoryImpl extends JdbcDaoSupport implements ContextRepository {
    private final String env;

    private final String ctxTable;

    private final String logTable;

    public ContextRepositoryImpl(String env, DataSource dataSource, String ctxTable, String logTable) {
        super.setDataSource(dataSource);
        this.env = env;
        this.ctxTable = ctxTable;
        this.logTable = logTable;
    }

    @Override
    public void saveContext(Context ctx, FireContext fire) {
        ContextEntity bean = ContextHelper.toSaveContext(ctx);
        if (this.isExistContext(ctx.getMachineId(), ctx.getTradeNo())) {
            super.getJdbcTemplate().update(
                SQL_UPDATE_CONTEXT_STATUS(this.ctxTable),
                bean.getCtxState(), bean.getRegionStates(), bean.getContext(), bean.getErrors(), bean.getSwitcher(),
                ctx.getLockExpireSecond(),
                ctx.getTradeNo(), ctx.getMachineId(), this.env);
        } else {
            super.getJdbcTemplate().update(
                SQL_INSERT_CONTEXT_STATUS(this.ctxTable),
                ctx.getMachineId(), ctx.getTradeNo(),
                bean.getCtxState(), bean.getRegionStates(), bean.getContext(), bean.getErrors(), bean.getSwitcher(),
                ctx.getLockVersion(), this.env, ctx.getLockExpireSecond());
        }
        super.getJdbcTemplate().update(SQL_INSERT_CONTEXT_LOG(this.logTable),
            ctx.getMachineId(), ctx.getTradeNo(),
            bean.getCtxState(), bean.getRegionStates(), bean.getContext(), bean.getErrors(), bean.getSwitcher(),
            fire.getFireEvent(), fire.getSourceState(), fire.getTargetState(), this.env);
    }

    @Override
    public boolean lock(Context ctx, String lockVersion) {
        boolean exists = this.isExistContext(ctx.getMachineId(), ctx.getTradeNo());
        if (exists) {
            return updateLock(ctx, lockVersion, ctx.getLockExpireSecond());
        } else {
            return true;
        }
    }

    @Override
    public boolean unlock(Context ctx) {
        return updateLock(ctx, "0", 0);
    }

    /**
     * 更新全局锁
     *
     * @param ctx           上下文
     * @param newLock       新的版本锁
     * @param expireSeconds 版本锁过期时间
     * @return
     */
    private boolean updateLock(Context ctx, String newLock, long expireSeconds) {
        int result = super.getJdbcTemplate().update(
            SQL_UPDATE_CONTEXT_LOCK(this.ctxTable),
            newLock, expireSeconds,
            ctx.getTradeNo(), ctx.getMachineId(), this.env, ctx.getLockVersion());
        if (result > 0) {
            ctx.setLockVersion(newLock);
            return true;
        } else {
            return false;
        }
    }

    @Override
    public boolean isExistContext(String machineId, String tradeNo) {
        List<Map<String, Object>> list = super.getJdbcTemplate()
            .queryForList(SQL_IS_EXISTS_CONTEXT(this.ctxTable), machineId, tradeNo, this.env);
        return list.size() > 0;
    }

    @Override
    public <DATA> Context<DATA> loadContext(String machineId, String tradeNo, Class<DATA> klass) {
        List<Map<String, Object>> list = super.getJdbcTemplate().queryForList(
            SQL_FIND_CONTEXT_BY_TRADE_NO(this.ctxTable), machineId, tradeNo, this.env);
        Context<DATA> ctx = list.size() == 0 ? null : ContextHelper.toContext(list.get(0), klass);
        return ctx;
    }

    @Override
    public Map<Long, String> findTradesByStatus(String machineId, String stateId, long id, int limit) {
        List<Map<String, Object>> list = super.getJdbcTemplate().queryForList(
            SQL_FIND_CONTEXT_BY_STATUS(this.ctxTable),
            machineId, this.env, stateId, id, limit);
        return list.stream().collect(toMap(m -> Long.parseLong(String.valueOf(m.get(F_ID))), m -> (String) m.get(F_TRADE_NO)));
    }
}