package cn.org.atool.fluentmachine.persistence;

import cn.org.atool.fluentmachine.context.Context;
import cn.org.atool.fluentmachine.context.StatusMap;
import com.google.gson.Gson;

import java.util.HashMap;
import java.util.Map;

import static cn.org.atool.fluentmachine.persistence.ContextRepositorySql.*;

/**
 * ContextHelper: 上下文序列化帮助类
 *
 * @author darui.wu
 */
public class ContextHelper {
    /**
     * 数据库结构转换为Context值
     *
     * @param map    记录行
     * @param klass  Context泛型对象
     * @param <DATA>
     * @return
     */
    public static <DATA> Context<DATA> toContext(Map<String, Object> map, Class<DATA> klass) {
        Context<DATA> ctx = new Context();

        ctx.setMachineId((String) map.get(F_MACHINE_ID));
        ctx.setTradeNo((String) map.get(F_TRADE_NO));
        ctx.setStateId((String) map.get(F_CTX_STATE));

        StatusMap status = stringToStatus((String) map.get(F_REGION_STATES));
        ctx.setStates(status);

        DATA data = stringToData((String) map.get(F_CONTEXT), klass);
        ctx.setData(data);
        Map<String, String> errors = gson.fromJson((String) map.get(F_ERRORS), HashMap.class);
        ctx.setErrors(errors);
        Map<String, String> switcher = gson.fromJson((String) map.get(F_SWITCHER), HashMap.class);
        ctx.setSwitcher(switcher);

        return ctx;
    }

    /**
     * 通过序列化和反序列化进行上下文深度拷贝
     *
     * @param source
     * @return
     */
    public static <DATA> Context<DATA> copy(Context<DATA> source) {
        Context<DATA> target = new Context<>();

        target.setMachineId(source.getMachineId());
        target.setTradeNo(source.getTradeNo());
        target.setStateId(source.getStateId());

        target.setStates(stringToStatus(statusToString(source.getStates())));
        if (source.getData() == null) {
            target.setData(null);
        } else {
            target.setData((DATA) stringToData(dataToString(source.getData()), source.getData().getClass()));
        }
        target.setErrors(gson.fromJson(gson.toJson(source.getErrors()), HashMap.class));
        target.setSwitcher(gson.fromJson(gson.toJson(source.getSwitcher()), HashMap.class));
        return target;
    }

    /**
     * 上下文数据转换为数据库模型
     *
     * @param ctx
     * @return
     */
    public static ContextEntity toSaveContext(Context ctx) {
        ContextEntity bean = new ContextEntity();

        bean.setMachineId(ctx.getMachineId());
        bean.setTradeNo(ctx.getTradeNo());
        bean.setCtxState(ctx.getStateId());
        bean.setRegionStates(statusToString(ctx.getStates()));
        bean.setContext(dataToString(ctx.getData()));
        bean.setErrors(gson.toJson(ctx.getErrors()));
        bean.setSwitcher(gson.toJson(ctx.getSwitcher()));
        return bean;
    }

    private static final Gson gson = new Gson();

    /**
     * 上下文状态序列化
     *
     * @param regions
     * @return
     */
    public static String statusToString(StatusMap regions) {
        return gson.toJson(regions);
    }

    /**
     * 上下文状态反序列化
     *
     * @param json
     * @return
     */
    public static StatusMap stringToStatus(String json) {
        return gson.fromJson(json, StatusMap.class);
    }

    /**
     * 上下文数据序列化
     *
     * @param data
     * @return
     */
    public static String dataToString(Object data) {
        return gson.toJson(data);
    }

    /**
     * 上下文反序列化
     *
     * @param json
     * @param klass
     * @param <DATA>
     * @return
     */
    public static <DATA> DATA stringToData(String json, Class<DATA> klass) {
        return gson.fromJson(json, klass);
    }
}