package com.github.aoudiamoncef.apollo.plugin.config

import com.apollographql.apollo.api.ApolloExperimental
import com.apollographql.apollo.compiler.NullableValueType
import com.apollographql.apollo.compiler.OperationIdGenerator
import java.io.File

/**
 * CompilerParams contains all the parameters needed to invoke the apollo compiler.
 *
 * The setters are present for backward compatibility with kotlin build scripts and will go away
 * in a future release.
 */
class CompilerParams {

    /**
     * Whether to generate Java or Kotlin models
     *
     * Default value: false
     */
    internal val generateKotlinModels: Boolean = false

    /**
     * Warn if using a deprecated field
     *
     * Default value: true
     */
    internal val warnOnDeprecatedUsages: Boolean = true

    /**
     * Fail the build if there are warnings. This is not named `allWarningAsErrors` to avoid nameclashes with the Kotlin options
     *
     * Default value: false
     */
    internal val failOnWarnings: Boolean = false

    /**
     * Whether to generate operationOutput.json. operationOutput.json contains information such as
     * operation id, name and complete source sent to the server. This can be used to upload
     * a query's exact content to a server that doesn't support automatic persisted queries.
     *
     * The operation output is written in [CompilationUnit.operationOutputFile]
     *
     * Default value: false
     */
    internal val generateOperationOutput: Boolean = false

    /**
     * For custom scalar types like Date, map from the GraphQL type to the jvm/kotlin type.
     *
     * Default value: the empty map
     */
    internal val customTypeMapping: Map<String, String> = emptyMap()

    /**
     * By default, Apollo uses `Sha256` hashing algorithm to generate an ID for the query.
     * To provide a custom ID generation logic, pass an `instance` that implements the [OperationIdGenerator]. How the ID is generated is
     * indifferent to the compiler. It can be an hashing algorithm or generated by a backend.
     * Default value: [OperationIdGenerator.Sha256]
     */
    internal val operationIdGeneratorClass: String = ""

    /**
     * The custom types code generate some warnings that might make the build fail.
     * suppressRawTypesWarning will add the appropriate SuppressWarning annotation
     *
     * Default value: false
     */
    internal val suppressRawTypesWarning: Boolean = false

    /**
     * When true, the generated classes names will end with 'Query' or 'Mutation'.
     * If you write `query droid { ... }`, the generated class will be named 'DroidQuery'.
     *
     * Default value: true
     */
    internal val useSemanticNaming: Boolean = true

    /**
     * The nullable value type to use. One of: "annotated", "apolloOptional", "guavaOptional", "javaOptional", "inputType"
     *
     * Default value: "annotated"
     * Only valid for java models as kotlin has proper nullability support
     */
    internal val nullableValueType: NullableValueType = NullableValueType.ANNOTATED

    /**
     * Whether to generate builders for java models
     *
     * Default value: false
     * Only valid for java models as kotlin has data classes
     */
    internal val generateModelBuilder: Boolean = false

    /**
     * When true, java beans getters and setters will be generated for fields. If you request a field named 'user', the generated
     * model will have a `getUser()` property instead of `user()`
     *
     * Default value: false
     * Only valid for java as kotlin has properties
     */
    internal val useJavaBeansSemanticNaming: Boolean = false

    /**
     * Apollo Maven plugin supports generating visitors for compile-time safe handling of polymorphic datatypes.
     * Enabling this requires source/target compatibility with Java 1.8.
     *
     * Default value: false
     */
    internal val generateVisitorForPolymorphicDatatypes: Boolean = false

    /**
     * The package name of the models is computed from their folder hierarchy like for java sources.
     *
     * If you want, you can prepend a custom package name here to namespace your models.
     *
     * Default value: the empty string
     */
    internal var rootPackageName: String = ""

    /**
     * Whether to generate Kotlin models with `internal` visibility modifier.
     *
     * Default value: false
     */
    internal val generateAsInternal: Boolean = false

    /**
     * A set of [Regex] patterns for GraphQL enums that should be generated as Kotlin sealed classes instead of the default Kotlin enums.
     *
     * Use this if you want your client to have access to the rawValue of the enum. This can be useful if new GraphQL enums are added but
     * the client was compiled against an older schema that doesn't have knowledge of the new enums.
     */
    internal val sealedClassesForEnumsMatching: Set<String> = emptySet()

    /**
     * Whether or not to generate Apollo metadata. Apollo metadata is used for multi-module support. Set this to true if you want other
     * modules to be able to re-use fragments and types from this module.
     *
     * This is currently experimental and this API might change in the future.
     *
     * Default value: false
     */
    @ApolloExperimental
    internal val generateApolloMetadata: Boolean = false

    /**
     * A list of [Regex] patterns for input/scalar/enum types that should be generated whether or not they are used by queries/fragments
     * in this module. When using multiple modules, Apollo Android will generate all the types by default in the root module
     * because the root module doesn't know what types are going to be used by dependent modules. This can be prohibitive in terms
     * of compilation speed for large projects. If that's the case, opt-in the types that are used by multiple dependent modules here.
     * You don't need to add types that are used by a single dependent module.
     *
     * This is currently experimental and this API might change in the future.
     *
     * Default value: if (generateApolloMetadata) listOf(".*") else listOf()
     */
    @ApolloExperimental
    internal var alwaysGenerateTypesMatching: Set<String> = setOf()

    /**
     * Use the given package name and does not take into account the folders hierarchy.
     *
     * - Operations will be in `packageName`
     * - Fragments will be in `packageName.fragment`
     * - Input/Enum/Scalar types are not handled yet and will continue to be in the schemaPackageName
     *
     * This is currently experimental and this API might change in the future.
     *
     * Default value: ""
     */
    @ApolloExperimental
    internal var packageName: String? = null

    /**
     * The rootFolders where the graphqlFiles are located. The package name of each individual graphql query
     * will be the relative path to the root folders
     */
    internal var rootFolders: List<File> = emptyList()

    /**
     * A list of files containing metadata from previous compilations
     */
    @ApolloExperimental
    internal val metadata: List<File> = emptyList()

    /**
     * The moduleName for this metadata. Used for debugging purposes
     */
    internal val moduleName: String = "?"

    /**
     * Optional rootProjectDir. If it exists:
     * - when writing metadata the compiler will output relative path to rootProjectDir
     * - when reading metadata the compiler will lookup the actual file
     * This allows to lookup the real fragment file if all compilation units belong to the same project
     * and output nicer error messages
     */
    internal val rootProjectDir: File? = null

    /**
     * The file where to write the metadata
     */
    internal var metadataOutputFile: File? = null

    /**
     * Whether the project is Kotlin multi-platform
     */
    internal val kotlinMultiPlatformProject: Boolean = false
}
