package com.github.antelopeframework.dynamicproperty.zk;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.lang3.StringUtils;

import com.github.antelopeframework.dynamicproperty.AbstractDynamicPropertyManager;
import com.github.antelopeframework.dynamicproperty.DynamicPropertyChangeListener;
import com.github.antelopeframework.dynamicproperty.WatchedUpdateResult;
import com.netflix.config.ConfigurationManager;
import com.netflix.config.DynamicPropertyFactory;
import com.netflix.config.DynamicStringProperty;
import com.netflix.config.DynamicWatchedConfiguration;

import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class ZkDynamicPropertyManager extends AbstractDynamicPropertyManager {
	private ConcurrentMap<String, DynamicStringProperty> dynamicProperties = new ConcurrentHashMap<>();

	@Getter
    private final ExecutorService threadExecutor = Executors.newCachedThreadPool();
	
	private final ZkConfigurationSource zkConfigurationSource; 
	private final DynamicPropertyFactory dynamicPropertyFactory;

	public ZkDynamicPropertyManager(ZkConfigurationSource zkConfigurationSource) {
		DynamicWatchedConfiguration zkDynamicConfig = new DynamicWatchedConfiguration(zkConfigurationSource);
		ConfigurationManager.install(zkDynamicConfig);
		
		this.zkConfigurationSource = zkConfigurationSource;
		this.dynamicPropertyFactory = DynamicPropertyFactory.getInstance();
	}

	@Override
	public void updateValue(String propName, String updatedValue) {
		try {
			zkConfigurationSource.setProperty(propName, updatedValue);
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		}
	}
	
	@Override
	public Map<String, Object> getAllProperties() {
		try {
			return zkConfigurationSource.getCurrentData();
		} catch (Exception e) {
			log.error(e.getMessage(), e);
		}
		
		return null;
	}
	
	@Override
	protected String getDynamicValue(String propName) {
		return dynamicPropertyFactory.getStringProperty(propName, null).get();
	}
	
	@Override
	public synchronized void addChangeListener(String propName, DynamicPropertyChangeListener listener) {
		super.addChangeListener(propName, listener);
		
		if (StringUtils.isBlank(propName) || listener == null) {
			return;
		}
		
		if (!dynamicProperties.containsKey(propName)) {
			DynamicStringProperty dynamicProperty = dynamicPropertyFactory.getStringProperty(propName, null);
			dynamicProperty.addCallback(new PropertyChangeCallback(propName, dynamicProperty, this));
			dynamicProperties.putIfAbsent(propName, dynamicProperty);
		}
	}

	private static class PropertyChangeCallback implements Runnable {
		private final String propName;
		private final ZkDynamicPropertyManager dynamicPropertyManager;
		private final DynamicStringProperty dynamicProperty;
		
		public PropertyChangeCallback(String propName, DynamicStringProperty dynamicProperty, ZkDynamicPropertyManager dynamicPropertyManager) {
			this.propName = propName;
			this.dynamicPropertyManager = dynamicPropertyManager;
			this.dynamicProperty = dynamicProperty;
		}
		
		@Override
		public void run() {
			String dValue = dynamicProperty.get();
			
			if (log.isDebugEnabled()) {
				log.debug("dynamic property change: key={}, value={}", propName, dValue);
			}
			
			//~ 不处理动态参数被删除的情况
			Map<String, Object> added = new HashMap<String, Object>();
            added.put(propName, dValue);
            
            final WatchedUpdateResult updateResult = WatchedUpdateResult.createIncremental(added, null, null);
            
            dynamicPropertyManager.getThreadExecutor().submit(new Callable<Boolean>() {

                @Override
                public Boolean call() throws Exception {
                	dynamicPropertyManager.fireEvent(updateResult);
                    return Boolean.TRUE;
                }
            });
			
		}
	}
}
