package com.github.antelopeframework.dynamicproperty.spring;

import java.util.Formatter;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.PropertyPlaceholderHelper;
import org.springframework.util.PropertyPlaceholderHelper.PlaceholderResolver;
import org.springframework.util.StringValueResolver;

import com.github.antelopeframework.dynamicproperty.DynamicPropertyChangeListener;
import com.github.antelopeframework.dynamicproperty.DynamicPropertyManager;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class GlobalVars implements InitializingBean {
    @Setter
    private boolean useTrim = true;
    
    @Setter
    @Autowired(required = true)
    private PropertyPlaceholderConfigurer propertyPlaceholderConfigurer;
    
    @Getter
    @Setter
    @Autowired(required = false)
    private DynamicPropertyManager dynamicPropertyManager;
    
    private StringValueResolver valueResolver;
    
    @Override
    public void afterPropertiesSet() throws Exception {
        Properties properties = propertyPlaceholderConfigurer.getProperties();
        valueResolver = new PlaceholderResolvingStringValueResolver(properties);
        
        if (log.isInfoEnabled()) {
            String configedProperties = buildPropertiesLog("Configed Properties: ", properties);
            log.info(configedProperties);
        }
    }
    
    /**
     * 获取参数值. 有些从动态参数中获取; 如果从动态参数中未获取到取值, 从Spring配置的Properties中获取. <br/>
     * 是否获取到有效值是通过{@link StringUtils#isBlank(CharSequence)}来检查的.
     * 
     * @param key
     * @return
     */
    public String getValue(String key) {
        String strVal = null;
        
        //~ 优先从动态参数中获取
        if (dynamicPropertyManager != null) {
            strVal = dynamicPropertyManager.getCurrentValue(key);
        }
        
        if (propertyPlaceholderConfigurer == null) {
            return useTrim ? StringUtils.trim(strVal) : strVal;
        }
        
        //~ 从预先配置的参数中获取
        if (StringUtils.isBlank(strVal)) {
            Properties properties = propertyPlaceholderConfigurer.getProperties();
            strVal = properties.getProperty(key);
        }
        
        //~ 占位符替换
        String value = null;
        if (StringUtils.isNotBlank(strVal)) {
            value = valueResolver.resolveStringValue(strVal);
        }
        
        return useTrim ? StringUtils.trim(value) : value;
    }
    
    /**
     * get {@link String}.
     * 
     * @param key
     * @param defaultValue
     * @return
     */
    public String getValue(String key, String defaultValue) {
        String value = getValue(key);
        
        return StringUtils.isBlank(value) ? defaultValue : value;
    }

    /**
     * get {@link Integer}.
     * 
     * @param key
     * @return
     */
    public Integer getInteger(String key) {
        String v = getValue(key);
        if (StringUtils.isBlank(v)) {
            return null;
        }
        
        return Integer.parseInt(v);
    }
    
    /**
     * get {@link Integer}.
     * 
     * @param key
     * @param defaultValue
     * @return
     */
    public Integer getInteger(String key, Integer defaultValue) {
        Integer v = null; 
        try {
            v = getInteger(key);
        } catch (Exception e) {
            //DO-NOTHING
        }
        
        if (v == null) {
            v = defaultValue;
        }
        
        return v;
    }
    
    /**
     * get {@link Long}.
     * 
     * @param key
     * @return
     */
    public Long getLong(String key) {
        String v = getValue(key);
        if (StringUtils.isBlank(v)) {
            return null;
        }
        
        return Long.parseLong(v);
    }
    
    /**
     * get {@link Long}.
     * 
     * @param key
     * @param defaultValue
     * @return
     */
    public Long getLong(String key, Long defaultValue) {
        Long v = null; 
        try {
            v = getLong(key);
        } catch (Exception e) {
            //DO-NOTHING
        }
        
        if (v == null) {
            v = defaultValue;
        }
        
        return v;
    }
    
    /**
     * get {@link Double}.
     * 
     * @param key
     * @return
     */
    public Double getDouble(String key) {
        String v = getValue(key);
        if (StringUtils.isBlank(v)) {
            return null;
        }
        
        return Double.parseDouble(v);
    }
    
    /**
     * get {@link Double}.
     * 
     * @param key
     * @param defaultValue
     * @return
     */
    public Double getDouble(String key, Double defaultValue) {
        Double v = null; 
        try {
            v = getDouble(key);
        } catch (Exception e) {
            //DO-NOTHING
        }
        
        if (v == null) {
            v = defaultValue;
        }
        
        return v;
    }
    
    /**
     * get {@link Number}.
     * 
     * @param key
     * @return
     */
    public Number getNumber(String key) {
        String v = getValue(key);
        if (StringUtils.isBlank(v)) {
            return null;
        }
        
        return NumberUtils.createNumber(v);
    }
    
    /**
     * get {@link Boolean}.
     * 
     * @param key
     * @return
     */
    public Boolean getBoolean(String key) {
        String v = getValue(key);
        if (StringUtils.isBlank(v)) {
            return null;
        }
        
        if ("yes".equals(v) || "y".equals(v) || "true".equals(v)) {
            return Boolean.TRUE;
        }
        
        if ("not".equals(v) || "n".equals(v) || "false".equals(v)) {
            return Boolean.FALSE;
        }
        
        throw new RuntimeException(String.format("Unsupport boolean value '%s'; only support 'yes/y/true' or 'not/n/false'.", v));
    }
    
    /**
     * get {@link Boolean}.
     * 
     * @param key
     * @return
     */
    public Boolean getBoolean(String key, Boolean defaultValue) {
        try {
            Boolean v = getBoolean(key);
            if (v == null) {
                return defaultValue;
            }
            
            return v;
        } catch (Exception e) {
            return defaultValue;
        }
    }
    
    /**
     * {@link org.springframework.beans.factory.config.PropertyPlaceholderConfigurer.PlaceholderResolvingStringValueResolver}
     */
    private class PlaceholderResolvingStringValueResolver implements StringValueResolver {
        private final PropertyPlaceholderHelper helper;
        private final PlaceholderResolver resolver;

        public PlaceholderResolvingStringValueResolver(Properties props) {
            this.helper = new PropertyPlaceholderHelper(
                    propertyPlaceholderConfigurer.getPlaceholderPrefix(), 
                    propertyPlaceholderConfigurer.getPlaceholderSuffix(), 
                    propertyPlaceholderConfigurer.getValueSeparator(), 
                    propertyPlaceholderConfigurer.getIgnoreUnresolvablePlaceholders());
            
            this.resolver = new PropertyPlaceholderConfigurerResolver(props);
        }

        @Override
        public String resolveStringValue(String strVal) throws BeansException {
            String value = this.helper.replacePlaceholders(strVal, this.resolver);
            return (value.equals(propertyPlaceholderConfigurer.getNullValue()) ? null : value);
        }
    }

    /**
     * {@link org.springframework.beans.factory.config.PropertyPlaceholderConfigurer.PropertyPlaceholderConfigurerResolver}
     */
    private class PropertyPlaceholderConfigurerResolver implements PlaceholderResolver {

        private final Properties props;

        private PropertyPlaceholderConfigurerResolver(Properties props) {
            this.props = props;
        }

        @Override
        public String resolvePlaceholder(String placeholderName) {
            return propertyPlaceholderConfigurer.resolvePlaceholder(placeholderName, props, PropertyPlaceholderConfigurer.SYSTEM_PROPERTIES_MODE_OVERRIDE);
        }
    }
    
    public String buildPropertiesLog(String prefix, Properties properties) {
        Formatter formatter = new Formatter();
        
        StringBuilder sb = new StringBuilder();
        sb.append("\n" + prefix + "\n");
        
        formatter.format("%32s %-16s\n", "-----key------", "--------value---------");
        for (Object key : properties.keySet()) {
            formatter.format("%32s %-16s\n", key, valueResolver.resolveStringValue(properties.getProperty(key.toString())));
        }
        
        sb.append(formatter.toString());
        formatter.close();
        
        return sb.toString();
    }
    
    /**
     * <p>注册动态属性值变化监听; 当<p>
     * <ul>
     *  <li>注册成功时: 返回true;</li>
     *  <li>注册失败时: 返回false.</li>
     * </ul>
     * <p>失败的场景包括: 参数值不齐备、{@link GlobalVars}实例中未设置{@link GlobalVars#dynamicPropertyManager}属性.</p>
     * @param globalVars
     * @param propName
     * @param listener
     * @return
     */
    public static boolean registDynamicPropertyListener(GlobalVars globalVars, String propName, DynamicPropertyChangeListener listener) {
        if (globalVars == null || StringUtils.isBlank(propName) || listener == null) {
            return false;
        }
        
        DynamicPropertyManager dpm = globalVars.getDynamicPropertyManager();
        if (dpm == null) {
            return false;
        }
        
        dpm.addChangeListener(propName, listener);

        return true;
    }
}
