package com.github.antelopeframework.dynamicproperty;

import java.util.Map;

import lombok.Getter;

/**
 * This class represents the result of a callback from the
 * WatchedConfigurationSource. The result may be the complete content of the
 * configuration source - or an incremental one.
 */
public class WatchedUpdateResult {
	@Getter
	protected final Map<String, Object> complete, added, changed, deleted;
	@Getter
	protected final boolean incremental;

	/**
	 * Create a full result that represents the complete content of the configuration source.
	 * 
	 * @param complete
	 * @return
	 */
	public static WatchedUpdateResult createFull(Map<String, Object> complete) {
		return new WatchedUpdateResult(complete);
	}

	/**
	 * 增量.
	 * 
	 * Create a result that represents incremental changes from the
	 * configuration source.
	 * 
	 * @param added properties added
	 * @param changed properties changed
	 * @param deleted properties deleted, in which case the value in the map will be ignored
	 */
	public static WatchedUpdateResult createIncremental(
			Map<String, Object> added, 
			Map<String, Object> changed, 
			Map<String, Object> deleted) {
		return new WatchedUpdateResult(added, changed, deleted);
	}

	/**
	 * Indicate whether this result has any content. If the result is
	 * incremental, this is true if there is any any added, changed or deleted
	 * properties. If the result is complete, this is true if
	 * {@link #getComplete()} is null.
	 */
	public boolean hasChanges() {
		if (incremental) {
			return (added != null && added.size() > 0) || (changed != null && changed.size() > 0) || (deleted != null && deleted.size() > 0);
		} else {
			return complete != null;
		}
	}

	WatchedUpdateResult(Map<String, Object> complete) {
		this.complete = complete;
		this.added = null;
		this.changed = null;
		this.deleted = null;
		this.incremental = false;
	}

	WatchedUpdateResult(Map<String, Object> added, Map<String, Object> changed, Map<String, Object> deleted) {
		this.complete = null;
		this.added = added;
		this.changed = changed;
		this.deleted = deleted;
		this.incremental = true;
	}
}