package com.github.antelopeframework.dynamicproperty;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.commons.lang3.StringUtils;

import lombok.Getter;

public abstract class AbstractDynamicPropertyManager implements DynamicPropertyManager {
    /** 用于对${@link #propertyChangeListeners}的修改操作 */
    private ReadWriteLock lock = new ReentrantReadWriteLock();
    
    @Getter
    private ConcurrentMap<String, List<DynamicPropertyChangeListener>> propertyChangeListeners = new ConcurrentHashMap<>();

    /**
     * 仅存储动态参数, 不缓存初始值
     */
    private final MapConfiguration configuration = new MapConfiguration(new Properties());
    
	@Override
	public void addChangeListener(String propName, DynamicPropertyChangeListener listener) {
		if (StringUtils.isBlank(propName) || listener == null) {
			return;
		}

		try {
			lock.writeLock().lock();

			List<DynamicPropertyChangeListener> listeners = propertyChangeListeners.get(propName);
			if (listeners == null) {
				propertyChangeListeners.putIfAbsent(propName, new ArrayList<DynamicPropertyChangeListener>());
				listeners = propertyChangeListeners.get(propName);
			}

			if (!listeners.contains(listener)) {
				listeners.add(listener);
			}

		} finally {
			lock.writeLock().unlock();
		}
	}

	@Override
	public void removeChangeListener(String propName, DynamicPropertyChangeListener listener) {
		if (listener == null) {
			return;
		}

		try {
			lock.writeLock().lock();

			List<DynamicPropertyChangeListener> listeners = propertyChangeListeners.get(propName);

			if (listeners != null) {
				listeners.remove(listener);
			}
		} finally {
			lock.writeLock().unlock();
		}
	}
    
    protected void fireEvent(WatchedUpdateResult updateResult) {
        DynamicPropertyUpdater.updateProperties(updateResult, configuration, false);
        
        if (!updateResult.isIncremental()) {
            Set<String> addProperties = new HashSet<String>();
            Map<String, Object> props = updateResult.getAdded();
            if (props != null) {
                addProperties.addAll(props.keySet());
            }
            
            if (addProperties.size() != 0) {
                triggerUpdateListener(addProperties);
            }
        } else {
            Set<String> updateProperties = new HashSet<String>();
            
            Map<String, Object> props = updateResult.getAdded();
            if (props != null) {
                updateProperties.addAll(props.keySet());
            }
            
            props = updateResult.getChanged();
            if (props != null) {
                updateProperties.addAll(props.keySet());
            }
            
            if (updateProperties.size() != 0) {
                triggerUpdateListener(updateProperties);
            }
            
            Set<String> deleteProperties = new HashSet<String>();
            
            props = updateResult.getDeleted();
            if (props != null) {
                deleteProperties.addAll(props.keySet());
            }
            
            if (deleteProperties.size() != 0) {
                for (String property : deleteProperties) {
                    List<DynamicPropertyChangeListener> listeners = propertyChangeListeners.get(property);
                    
                    if (listeners == null || listeners.size() == 0) {
                        continue;
                    }
                    
                    for (DynamicPropertyChangeListener listener : listeners) {
                        listener.onDelete(getCurrentValue(property));
                    }
                }
            }
        }
    }
    
    protected void triggerUpdateListener(Set<String> properties) {
        for (String property : properties) {
            List<DynamicPropertyChangeListener> listeners = propertyChangeListeners.get(property);
            if (listeners == null || listeners.size() == 0) {
                continue;
            }
            
            String value = getCurrentValue(property);
            for (DynamicPropertyChangeListener listener : listeners) {
                listener.onUpdate(value);
            }
        }
    }

    @Override
    public String getCurrentValue(String propName) {
    	if (!this.configuration.containsKey(propName)) {
    		String value = getDynamicValue(propName);
    		if (StringUtils.isNotBlank(value)) {
    			this.configuration.addProperty(propName, value);
    		}
    	}
    	
    	return this.configuration.getString(propName);
    }
    
    protected abstract String getDynamicValue(String propName);
}
