package com.github.tonywills.droidbase.ui;

import android.animation.Animator;
import android.annotation.TargetApi;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.os.Build;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewAnimationUtils;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.RelativeLayout;

import com.github.tonywills.droidbase.R;
import com.github.tonywills.droidbase.utils.SimpleAnimatorListener;
import com.github.tonywills.droidbase.utils.Utils;

public class ColourChangeButton extends FrameLayout {

    private int[] colours;
    private int currentColourIndex;

    private Button nextColourView;
    private Button currentColorView;

    private ColourChangeButtonWillChangeListener willChangeListener;
    private ColourChangeButtonDidChangeListener didChangeListener;

    public ColourChangeButton(Context context) {
        this(context, null);
    }

    public ColourChangeButton(Context context, AttributeSet attrs) {
        super(context, attrs);
        RelativeLayout.inflate(context, R.layout.button_colour_change, this);
        onFinishInflate();
        pullXmlArgs(context, attrs);
    }

    public ColourChangeButton(Context context, AttributeSet attrs, int defStyleAttr) {
        this(context, attrs);
    }

    public ColourChangeButton(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        this(context, attrs);
    }

    public void setWillChangeListener(ColourChangeButtonWillChangeListener willChangeListener) {
        this.willChangeListener = willChangeListener;
    }

    public void setDidChangeListener(ColourChangeButtonDidChangeListener didChangeListener) {
        this.didChangeListener = didChangeListener;
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        nextColourView = (Button) findViewById(R.id.next_button);
        currentColorView = (Button) findViewById(R.id.current_button);
        nextColourView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                callOnClick();
            }
        });
        currentColorView.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                callOnClick();
            }
        });
    }

    private void pullXmlArgs(Context context, AttributeSet attrs) {
        TypedArray array = context.obtainStyledAttributes(attrs, R.styleable.ColourChangeButton);

        int colourArrayResourceId = array.getResourceId(R.styleable.ColourChangeButton_colours, -1);
        if (colourArrayResourceId != -1) {
            colours = getResources().getIntArray(colourArrayResourceId);
        }
        if (colours != null && colours.length > 0) {
            setBackgroundColourForButton(currentColorView, colours[0]);
            if (colours.length > 1) {
                setBackgroundColourForButton(nextColourView, colours[1]);
            }
        }
        setText(array.getString(R.styleable.ColourChangeButton_buttonText));
        setTextColour(array.getColor(R.styleable.ColourChangeButton_buttonTextColour, Color.LTGRAY));

        array.recycle();
    }

    @Override
    public boolean callOnClick() {
        setCurrentColorIndex(currentColourIndex + 1);
        return super.callOnClick();
    }

    public void setPossibleColours(int[] colours) {
        this.colours = colours;
        if (currentColorView != null) {
            setBackgroundColourForButton(currentColorView, colours[0]);
        }
    }

    public void setText(CharSequence text) {
        if (currentColorView != null) {
            currentColorView.setText(text);
        }
        if (nextColourView != null) {
            nextColourView.setText(text);
        }
    }

    public void setTextColour(int colour) {
        if (currentColorView != null) {
            currentColorView.setTextColor(colour);
        }
        if (nextColourView != null) {
            nextColourView.setTextColor(colour);
        }
    }

    private void setCurrentColorIndex(int colourIndex) {
        if (colours != null) {
            int oldColourIndex = currentColourIndex;
            currentColourIndex = colourIndex % colours.length;
            notifyWillChangeColour();
            animateColourChange(oldColourIndex, currentColourIndex);
        }
    }

    public int getColour() {
        return getCurrentColour();
    }

    public void setTransientColour(int colour) {
        setBackgroundColourForButton(currentColorView, colour);
    }

    // MARK: Animate Colour Change

    private void animateColourChange(int oldColourIndex, final int newColourIndex) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            circularAnimateColourChange();
        } else {
            Utils.animateColour(colours[oldColourIndex], colours[newColourIndex], 300, new Utils.ColourAnimationListener() {
                @Override
                public void colourAnimationProgressedTo(int blended) {
                    currentColorView.setBackgroundColor(blended);
                    if (blended == colours[newColourIndex]) {
                        notifyDidChangeColour();
                    }
                }
            });
        }
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private void circularAnimateColourChange() {
        setBackgroundColourForButton(nextColourView, getCurrentColour());

        int width = nextColourView.getMeasuredWidth();
        int height = nextColourView.getMeasuredHeight();

        // define the circle
        // center
        int cx = (int) (width * 0.5);
        int cy = (int) (height * 0.9);

        // radius
        int finalRadius = (int) Math.sqrt(
                Math.pow(width  * 0.5, 2) +
                Math.pow(height * 0.9, 2)
        );

        // create the animator for this view (the start radius is zero)
        Animator anim = ViewAnimationUtils.createCircularReveal(nextColourView, cx, cy, 0, finalRadius);
        anim.setDuration(300);

        // make the view visible and start the animation
        nextColourView.setVisibility(View.VISIBLE);
        anim.addListener(new SimpleAnimatorListener() {
            @Override
            public void onAnimationEnd(Animator animation) {
                setBackgroundColourForButton(currentColorView, getCurrentColour());
                nextColourView.setVisibility(INVISIBLE);
                notifyDidChangeColour();
            }
        });
        anim.start();
    }

    private void setBackgroundColourForButton(Button button, int colour) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            button.getBackground().setTint(colour);
        } else {
            //noinspection deprecation
            button.setBackgroundDrawable(Utils.getColorDrawableFromColor(colour));
        }
    }

    // MARK: Access Helpers

    private int getCurrentColour() {
        return colours[currentColourIndex];
    }

    private int getNextColour() {
        return colours[(currentColourIndex + 1) % colours.length];
    }

    // MARK: Notify Listeners of Events

    private void notifyDidChangeColour() {
        if (didChangeListener != null) {
            didChangeListener.didChangeColour(ColourChangeButton.this, getCurrentColour());
        }
    }

    private void notifyWillChangeColour() {
        if (willChangeListener != null) {
            willChangeListener.willChangeColour(ColourChangeButton.this, getCurrentColour());
        }
    }

    public interface ColourChangeButtonDidChangeListener {
        void didChangeColour(ColourChangeButton button, int colour);
    }

    public interface ColourChangeButtonWillChangeListener {
        void willChangeColour(ColourChangeButton button, int colour);
    }

}

