/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.EventDebouncer;
import com.datastax.driver.core.MemoryAppender;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.log4j.Appender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.assertj.core.api.Assertions;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

public class EventDebouncerTest {
    private ScheduledExecutorService executor;
    private MockDeliveryCallback callback;

    @BeforeMethod(groups={"unit"})
    public void setup() {
        this.executor = Executors.newScheduledThreadPool(1);
        this.callback = new MockDeliveryCallback();
    }

    @AfterMethod(groups={"unit"})
    public void tearDown() {
        this.executor.shutdownNow();
    }

    @Test(groups={"unit"})
    public void should_deliver_single_event() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 10;
            }

            long delayMs() {
                return 50L;
            }
        };
        debouncer.start();
        MockEvent event = new MockEvent(0);
        debouncer.eventReceived((Object)event);
        this.callback.awaitEvents(1);
        Assertions.assertThat(this.callback.getEvents()).containsOnly((Object[])new MockEvent[]{event});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(groups={"unit"})
    public void should_log_and_drop_events_on_overflow() throws InterruptedException {
        MemoryAppender logs = new MemoryAppender();
        Logger logger = Logger.getLogger(EventDebouncer.class);
        Level originalLoggerLevel = logger.getLevel();
        logger.setLevel(Level.WARN);
        logger.addAppender((Appender)logs);
        try {
            EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback, 10){

                int maxPendingEvents() {
                    return 100;
                }

                long delayMs() {
                    return 15L;
                }
            };
            debouncer.start();
            ArrayList<MockEvent> events = new ArrayList<MockEvent>();
            for (int i = 0; i < 14; ++i) {
                MockEvent event = new MockEvent(i);
                events.add(event);
                debouncer.eventReceived((Object)event);
            }
            this.callback.awaitEvents(10);
            Assertions.assertThat(this.callback.getEvents()).isEqualTo(events.subList(0, 10));
            Assertions.assertThat((String)logs.get()).containsOnlyOnce((CharSequence)"test debouncer enqueued more than 10 events, rejecting new events.");
        }
        finally {
            logger.removeAppender((Appender)logs);
            logger.setLevel(originalLoggerLevel);
        }
    }

    @Test(groups={"unit"})
    public void should_deliver_n_events_in_order() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 10;
            }

            long delayMs() {
                return 50L;
            }
        };
        debouncer.start();
        ArrayList<MockEvent> events = new ArrayList<MockEvent>();
        for (int i = 0; i < 50; ++i) {
            MockEvent event = new MockEvent(i);
            events.add(event);
            debouncer.eventReceived((Object)event);
        }
        this.callback.awaitEvents(50);
        Assertions.assertThat(this.callback.getEvents()).isEqualTo(events);
    }

    @Test(groups={"unit"})
    public void should_deliver_n_events_in_order_even_if_queue_full() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 10;
            }

            long delayMs() {
                return 1L;
            }
        };
        debouncer.start();
        ArrayList<MockEvent> events = new ArrayList<MockEvent>();
        for (int i = 0; i < 50; ++i) {
            MockEvent event = new MockEvent(i);
            events.add(event);
            debouncer.eventReceived((Object)event);
        }
        this.callback.awaitEvents(50);
        Assertions.assertThat(this.callback.getEvents()).isEqualTo(events);
    }

    @Test(groups={"unit"})
    public void should_accumulate_events_if_not_ready() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 10;
            }

            long delayMs() {
                return 50L;
            }
        };
        ArrayList<MockEvent> events = new ArrayList<MockEvent>();
        for (int i = 0; i < 50; ++i) {
            MockEvent event = new MockEvent(i);
            events.add(event);
            debouncer.eventReceived((Object)event);
        }
        debouncer.start();
        this.callback.awaitEvents(50);
        Assertions.assertThat(this.callback.getEvents()).hasSize(50);
        Assertions.assertThat(this.callback.getEvents()).isEqualTo(events);
    }

    @Test(groups={"unit"})
    public void should_accumulate_all_events_until_start() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 10;
            }

            long delayMs() {
                return 25L;
            }
        };
        ArrayList<MockEvent> events = new ArrayList<MockEvent>();
        for (int i = 0; i < 50; ++i) {
            MockEvent event = new MockEvent(i);
            events.add(event);
            debouncer.eventReceived((Object)event);
        }
        debouncer.start();
        this.callback.awaitEvents(50);
        Assertions.assertThat(this.callback.getEvents()).isEqualTo(events);
    }

    @Test(groups={"unit"})
    public void should_reset_timer_if_n_events_received_within_same_window() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 50;
            }

            long delayMs() {
                return 50L;
            }
        };
        debouncer.start();
        final CountDownLatch latch = new CountDownLatch(50);
        ScheduledExecutorService pool = Executors.newScheduledThreadPool(1);
        pool.scheduleAtFixedRate(new Runnable((EventDebouncer)debouncer){
            final /* synthetic */ EventDebouncer val$debouncer;
            {
                this.val$debouncer = eventDebouncer;
            }

            @Override
            public void run() {
                if (latch.getCount() > 0L) {
                    MockEvent event = new MockEvent(0);
                    this.val$debouncer.eventReceived((Object)event);
                    latch.countDown();
                }
            }
        }, 0L, 5L, TimeUnit.MILLISECONDS);
        latch.await();
        pool.shutdownNow();
        this.callback.awaitEvents(50);
        Assertions.assertThat(this.callback.getEvents()).hasSize(50);
    }

    @Test(groups={"unit"})
    public void should_stop_receiving_events() throws InterruptedException {
        EventDebouncer<MockEvent> debouncer = new EventDebouncer<MockEvent>("test", this.executor, (EventDebouncer.DeliveryCallback)this.callback){

            int maxPendingEvents() {
                return 10;
            }

            long delayMs() {
                return 50L;
            }
        };
        debouncer.start();
        for (int i = 0; i < 50; ++i) {
            MockEvent event = new MockEvent(i);
            debouncer.eventReceived((Object)event);
        }
        this.callback.awaitEvents(50);
        debouncer.stop();
        MockEvent event = new MockEvent(0);
        debouncer.eventReceived((Object)event);
        Assertions.assertThat(this.callback.getEvents()).hasSize(50);
    }

    private class MockEvent {
        private final int i;

        private MockEvent(int i) {
            this.i = i;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            MockEvent mockEvent = (MockEvent)o;
            return this.i == mockEvent.i;
        }

        public int hashCode() {
            return this.i;
        }

        public String toString() {
            return "MockEvent" + this.i;
        }
    }

    private static class MockDeliveryCallback
    implements EventDebouncer.DeliveryCallback<MockEvent> {
        final List<MockEvent> events = new CopyOnWriteArrayList<MockEvent>();
        final Lock lock = new ReentrantLock();
        final Condition cond = this.lock.newCondition();

        private MockDeliveryCallback() {
        }

        public ListenableFuture<?> deliver(List<MockEvent> events) {
            this.lock.lock();
            try {
                this.events.addAll(events);
                this.cond.signal();
            }
            finally {
                this.lock.unlock();
            }
            return Futures.immediateFuture(null);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void awaitEvents(int expected) throws InterruptedException {
            long nanos = TimeUnit.MINUTES.toNanos(5L);
            this.lock.lock();
            try {
                while (this.events.size() < expected) {
                    if (nanos <= 0L) {
                        Assertions.fail((String)"Timed out waiting for events");
                    }
                    nanos = this.cond.awaitNanos(nanos);
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        public List<MockEvent> getEvents() {
            return this.events;
        }
    }
}

